package ru.yandex.direct.oneshot.oneshots.generatesmartcreatives

import java.util.function.Consumer
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.bannerstorage.client.BannerStorageClient
import ru.yandex.direct.bannerstorage.client.BannerStorageClientException
import ru.yandex.direct.bannerstorage.client.model.Creative
import ru.yandex.direct.bannerstorage.client.model.CreativeGroup
import ru.yandex.direct.core.entity.creative.model.CreativeType
import ru.yandex.direct.core.entity.creative.model.StatusModerate
import ru.yandex.direct.core.entity.creative.service.CreativeService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.utils.CollectionUtils
import ru.yandex.direct.utils.CommonUtils.nvl
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeField
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeObject
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes


data class UpdateParam(
    val ytCluster: YtCluster,
    val tablePath: String
)

data class UpdateState(
    val lastRow: Long = 0L
)

@YTreeObject
data class UpdateInputTableRow(
    @YTreeField(key = "client_id") val clientId: Long,
    @YTreeField(key = "creative_group_id") val creativeGroupId: Long,
)

/**
 * Ваншот обновления креативов
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers("buhter", "zakhar")
class UpdateCreativesOneshot(
    private val ytProvider: YtProvider,
    private val bannerStorageClient: BannerStorageClient,
    private val creativeService: CreativeService,
) : SimpleOneshot<UpdateParam, UpdateState?> {

    companion object {
        private val logger = LoggerFactory.getLogger(UpdateCreativesOneshot::class.java)
        private const val CHUNK_SIZE = BannerStorageClient.MAX_CREATIVES_IN_BATCH
    }

    override fun validate(inputData: UpdateParam): ValidationResult<UpdateParam, Defect<*>>? {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(ytProvider, vb, inputData.ytCluster, inputData.tablePath)
    }

    override fun execute(inputData: UpdateParam, prevState: UpdateState?): UpdateState? {
        val startRow = prevState?.lastRow ?: 0L
        val lastRow = startRow + CHUNK_SIZE
        logger.info("Start from row=$startRow, to row=$lastRow (excluded)")

        val items = readInputTable(inputData, startRow, lastRow)

        items.forEach { updateCreativeGroup(ClientId.fromLong(it.clientId), it.creativeGroupId.toInt()) }

        if (items.size < CHUNK_SIZE) {
            logger.info("Last iteration finished")
            return null
        }
        return UpdateState(lastRow)
    }

    private fun readInputTable(inputData: UpdateParam, startRow: Long, lastRow: Long): List<UpdateInputTableRow> {
        val entryType = YTableEntryTypes.yson(UpdateInputTableRow::class.java)
        val ytTable = YtTable(inputData.tablePath)

        val items = mutableListOf<UpdateInputTableRow>()
        ytProvider.get(inputData.ytCluster).tables()
            .read(ytTable.ypath().withRange(startRow, lastRow), entryType,
                Consumer { row -> items.add(row) })
        return items
    }

    private fun updateCreativeGroup(clientId: ClientId, creativeGroupId: Int) {
        val existingCreativeGroup = bannerStorageClient.getSmartCreativeGroup(creativeGroupId)
        val creatives = existingCreativeGroup.creatives.map {
            Creative()
                .withId(it.id)
                .withVersion(it.version)
                .withParameters(it.parameters) // параметры оставляем те же
                .withTnsArticles(it.tnsArticles)
                .withTnsBrands(it.tnsBrands)
        }

        val creativeGroup: CreativeGroup = try {
            bannerStorageClient.editSmartCreativeGroup(creativeGroupId,
                CreativeGroup(existingCreativeGroup.id, existingCreativeGroup.name, creatives))

        } catch (e: BannerStorageClientException) {
            logger.error("Failed to update creative group $creativeGroupId", e)
            return
        }
        logger.info("Creative group $creativeGroupId updated in BS")

        val bsCreativeById = creativeGroup.creatives.associateBy { it.id.toLong() }

        val coreCreatives = creativeService.get(clientId, bsCreativeById.keys, listOf(CreativeType.PERFORMANCE))
        if (coreCreatives.size != bsCreativeById.keys.size) {
            logger.warn("BannerStorage has more creatives in group $creativeGroupId, " +
                "bsIds: ${bsCreativeById.keys}, coreIds: ${coreCreatives.map { it.id }}")
        }

        coreCreatives.filter { it.id in bsCreativeById }
            .forEach {
                val bsCreative = bsCreativeById[it.id]!!
                it
                    .withName(bsCreative.name)
                    .withGroupName(creativeGroup.name)
                    .withWidth(bsCreative.width.toLong())
                    .withHeight(bsCreative.height.toLong())
                    .withPreviewUrl(nvl(bsCreative.screenshotUrl, bsCreative.thumbnailUrl))
                    .withLivePreviewUrl(bsCreative.preview.url)
                    .withVersion(bsCreative.version.toLong())
                    .withModerationInfo(null)
                    .withIsBannerstoragePredeployed(bsCreative.isPredeployed)
                    .withStatusModerate(StatusModerate.NEW)
            }

        val result: MassResult<Long> = creativeService.createOrUpdate(coreCreatives, clientId)
        if (!CollectionUtils.isEmpty(result.errors)) {
            logger.error("Failed to update creatives ${result.errors}")
        }
        logger.info("Update of $creativeGroupId done")
    }

}
