package ru.yandex.direct.oneshot.oneshots.hypergeo

import com.google.common.collect.Lists
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoSegmentRepository
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

// сколько объектов с одного шарда удалится при запуске ваншота
private const val IDS_TO_DELETE_LIMIT = 10_000

data class InputData(
    val deleteHyperGeos: Boolean = true,
    val deleteHyperGeoSegments: Boolean = true,
    val idsToDeleteByShardLimit: Int = IDS_TO_DELETE_LIMIT,
)

/**
 * Ваншот для удаления неиспользуемых гипер гео (как самих условий ретаргетинга/целей, так и сегментов Аудиторий).
 *
 * Неиспользуемыми являются те гипер гео, которые перестали использоваться во всех группах, но не были удалены
 * (в Аудиториях, где хранятся сегменты, есть ограничение на число неиспользуемых сегментов).
 * Такие гипер гео должны органично удаляться при сохранении изменений на группе (и дополнительно ежедневной джобой,
 * которая подбирает все упавшие кейсы), но для очистки уже накопленных изменений или для срочного удаления лишних
 * гипер гео будет использоваться данный ваншот.
 *
 * Ваншот принимает на вход булевские параметры, отвечающие за удаление гипер и/или сегментов
 * (как минимум один из этих типов объектов удалить нужно), и лимит числа id, которые нужно удалить на каждом шарде.
 *
 * Дефолтно (без входных параметров) пытается удалить до 10_000 id гипер гео и сегментов на каждом шарде.
 */
@Component
@Multilaunch
@Approvers("pavelkataykin", "dlyange")
class DeleteUnusedHyperGeosOneshot @Autowired constructor(
    private val hyperGeoService: HyperGeoService,
    private val hyperGeoRepository: HyperGeoRepository,
    private val hyperGeoSegmentRepository: HyperGeoSegmentRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) : ShardedOneshot<InputData, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(DeleteUnusedHyperGeosOneshot::class.java)

        // сколько объектов удалится за одну итерацию внутри ваншота
        private const val IDS_TO_DELETE_CHUNK = 1_000
    }

    // Проверяем, что нужно удалить как минимум один из типов объектов (гипер гео или сегменты Аудиторий)
    override fun validate(inputData: InputData) = validateObject(inputData) {
        if (!inputData.deleteHyperGeos) {
            property(inputData::deleteHyperGeoSegments)
                .check(CommonConstraints.isTrue())
        }
        property(inputData::idsToDeleteByShardLimit)
            .check(CommonConstraints.notNull())
            .check(NumberConstraints.notLessThan(1))
            .check(NumberConstraints.notGreaterThan(100_000))
    }

    override fun execute(inputData: InputData, prevState: Void?, shard: Int): Void? {
        if (inputData.deleteHyperGeos) {
            deleteUnusedHyperGeos(shard, inputData.idsToDeleteByShardLimit)
        }
        if (inputData.deleteHyperGeoSegments) {
            deleteUnusedHyperGeoSegments(shard, inputData.idsToDeleteByShardLimit)
        }

        return null
    }

    // удалить гипер гео (условия ретаргетинга + цели) и сегменты (в Аудиториях и в hypergeo_segments)
    private fun deleteUnusedHyperGeos(shard: Int, idsLimit: Int) {
        logger.info("shard $shard: start deleting unused hyper geos and their segments")
        val hyperGeoIds = hyperGeoRepository.getUnusedHyperGeoIds(shard, idsLimit)
        logger.info("shard $shard: ${hyperGeoIds.size} unused hyper geos (out of $idsLimit limit)")
        for (hyperGeoIdsChunk in Lists.partition(hyperGeoIds.toList(), IDS_TO_DELETE_CHUNK)) {
            logAndDeleteHyperGeos(shard, hyperGeoIdsChunk)
        }
        logger.info("shard $shard: finished deleting unused hyper geos and their segments")
    }

    private fun logAndDeleteHyperGeos(shard: Int, hyperGeoIds: List<Long>) {
        logger.info("shard $shard: trying to delete $hyperGeoIds hyper geos and their segments")
        val hyperGeosData = hyperGeoRepository.getHyperGeosToLog(shard, hyperGeoIds)
        hyperGeosData.forEach { logger.info("Trying to delete hyper geo $it") }

        hyperGeoService.deleteHyperGeos(shard, null, hyperGeoIds)

        val notDeletedHyperGeoIds = hyperGeoRepository.getHyperGeosToLog(shard, hyperGeoIds)
            .map { it.id }
            .toSet()
        val deletedHyperGeoIds = hyperGeoIds.minus(notDeletedHyperGeoIds)
        if (deletedHyperGeoIds.isNotEmpty()) {
            logger.info("shard $shard: successfully deleted $deletedHyperGeoIds hyper geos")
        }
        if (notDeletedHyperGeoIds.isNotEmpty()) {
            logger.info("shard $shard: unsuccessfully tried to delete $notDeletedHyperGeoIds hyper geos")
        }
    }

    // удалить сегменты (в Аудиториях и в hypergeo_segments)
    private fun deleteUnusedHyperGeoSegments(shard: Int, idsLimit: Int) {
        val idsNotToDeleteProperty: PpcProperty<Set<Long>> =
            ppcPropertiesSupport.get(PpcPropertyNames.HYPER_GEO_SEGMENT_IDS_TO_BYPASS_DELETING)
        val idsNotToDelete = idsNotToDeleteProperty.getOrDefault(setOf())

        logger.info("shard $shard: start deleting unused hyper geo segments")
        val hyperGeoSegmentIds = hyperGeoSegmentRepository.getUnusedHyperGeoSegmentIds(shard, idsNotToDelete, idsLimit)
        logger.info("shard $shard: ${hyperGeoSegmentIds.size} unused hyper geo segments")
        for (hyperGeoSegmentIdsChunk in Lists.partition(hyperGeoSegmentIds.toList(), IDS_TO_DELETE_CHUNK)) {
            logAndDeleteHyperGeoSegments(shard, hyperGeoSegmentIdsChunk)
        }
        logger.info("shard $shard: finished deleting unused hyper geo segments")
    }

    private fun logAndDeleteHyperGeoSegments(shard: Int, hyperGeoSegmentIds: List<Long>) {
        logger.info("shard $shard: trying to delete $hyperGeoSegmentIds hyper geo segments")
        val hyperGeosData = hyperGeoSegmentRepository.getHyperGeoSegmentsToLog(shard, hyperGeoSegmentIds)
        hyperGeosData.forEach { logger.info("Trying to delete hyper geo segment $it") }

        hyperGeoService.deleteHyperGeoSegments(shard, hyperGeoSegmentIds)

        val notDeletedSegmentIds = hyperGeoSegmentRepository.getHyperGeoSegmentsToLog(shard, hyperGeoSegmentIds)
            .map { it.id }
            .toSet()
        val deletedHyperGeoSegmentIds = hyperGeoSegmentIds.minus(notDeletedSegmentIds)
        if (deletedHyperGeoSegmentIds.isNotEmpty()) {
            logger.info("shard $shard: successfully deleted $deletedHyperGeoSegmentIds hyper geo segments")
        }
        if (notDeletedSegmentIds.isNotEmpty()) {
            logger.info("shard $shard: unsuccessfully tried to delete $notDeletedSegmentIds hyper geo segments")
        }
    }
}
