package ru.yandex.direct.oneshot.oneshots.importcashbackdata

import one.util.streamex.EntryStream
import one.util.streamex.StreamEx
import org.jooq.DSLContext
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_CASHBACK_HISTORY
import ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_CASHBACK_PROGRAMS
import ru.yandex.direct.dbschema.ppc.enums.ClientsCashbackHistoryStateChange
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsCashbackHistoryRecord
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsCashbackProgramsRecord
import ru.yandex.direct.dbschema.ppcdict.Tables
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.oneshot.oneshots.importcashbackdata.entity.YtCashbackClients
import ru.yandex.direct.oneshot.oneshots.importcashbackdata.entity.YtCashbackClientsRow
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.inside.yt.kosher.ytree.YTreeNode
import ru.yandex.misc.time.MoscowTime
import java.time.Instant
import java.time.LocalDateTime
import java.util.*
import java.util.stream.Collectors
import kotlin.streams.toList

data class CashbackClientsParam(
        val programIds: Set<Long>
)

fun getPrograms(dsl: DslContextProvider, programIds: Collection<Long>) =
        dsl.ppcdict().select(Tables.CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID)
                .from(Tables.CASHBACK_PROGRAMS)
                .where(Tables.CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID.`in`(programIds))
                .fetch().size

/**
 * Ваншот для разового применения.
 * Переносит данные о текущих клиентских установках кешбеков из трекера в нашу базу
 * В качестве параметра получает id заведенных у нас программ(3 штуки)
 * Не чанкует, так как данных не много — 50К строк
 */
@Component
@Approvers("kozobrodov", "pavryabov", "buhter")
class ImportCashbackDataOneshot @Autowired constructor(
        private val ytProvider: YtProvider,
        private val dsl: DslContextProvider,
        private val shardHelper: ShardHelper,
) : SimpleOneshot<CashbackClientsParam, Void> {
    data class CashbackClientInfo(
            var clientId: Long,
            val archived: Boolean,
            val added: LocalDateTime
    )

    private val ytCashbackClients = YtCashbackClients()
    private val statusArchived = "52566f1df2d224cc33a7f63e"
    private val componentsGoodLogin = "5ed064d6802a026dec232091"
    private val componentsGotInitMail = "5ede4c0950ef382990708364"
    private val componentsNotDuplicate = "5ed0665af004b0768e785c54"
    private val componentsNotOnHold = "5ed06ccf2d0bfa3c7723a0af"
    private val customFieldConferenceName = "conference"
    private val customFieldClientIdName = "cloudId"

    /**
     * Провереям, что переданные программы заведены у нас в базе
     */
    override fun validate(inputData: CashbackClientsParam): ValidationResult<CashbackClientsParam, Defect<*>> {
        return ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.programIds, "programIds")
                    .check(CommonConstraints.notNull())
                    .check(CollectionConstraints.notEmptyCollection())
                    .check(CollectionConstraints.setSize(getPrograms(dsl, inputData.programIds),
                            getPrograms(dsl, inputData.programIds)))
        }.result
    }

    override fun execute(inputData: CashbackClientsParam, prevState: Void?): Void? {
        //Читаем данные из YT таблицы
        val clientsToAdd = getClientsForCashback()
        //На основании полученных данных генерим записи и кладем их в базу
        addCashbackClientsWithHistory(inputData.programIds, clientsToAdd)
        return null
    }

    private fun addCashbackClientsWithHistory(programIds: Set<Long>,
                                              clientsToAdd: List<CashbackClientInfo>) {

        val clientIdToClients = clientsToAdd.associateBy({ it.clientId }, { it })
        val existingClientIds = shardHelper.getExistingClientIds(clientIdToClients.keys.toList())
        //Берем только ClientID, которые есть в базе
        val existingClientIdToClients = EntryStream.of(clientIdToClients)
                .filterKeys { existingClientIds[it] == true }
                .toMap()

        //Бьем по шардам
        shardHelper.groupByShard(existingClientIdToClients.values, ShardKey.CLIENT_ID) { it.clientId }
                .forEach { shard, cashbackClients ->
                    // Убираем возможные дубли по clientId, с приоритетом к отключенным от программы кешбеков
                    val filteredClients = StreamEx.of(cashbackClients)
                            .collapse({ t, u -> t.clientId == u.clientId }, { t, u -> if (t.archived) t else u })
                            .toList()

                    val clientsCashbackPrograms: MutableList<ClientsCashbackProgramsRecord> = ArrayList()
                    //Добавляем подключенные программы
                    for (programId in programIds) {
                        clientsCashbackPrograms.addAll(filteredClients
                                .stream()
                                .filter { client -> !client.archived }
                                .map { toClientsCashbackProgramsRecord(it, programId, 1) }
                                .toList())
                    }
                    //Добавляем вышедших из программ
                    clientsCashbackPrograms.addAll(filteredClients
                            .stream()
                            .filter { client -> client.archived }
                            .map { toClientsCashbackProgramsRecord(it, 1, 0) }
                            .toList())

                    //Пишем в базу связь клиент-программа и получаем обратно в т.ч. id записей
                    val clientPrograms = writeClientPrograms(dsl.ppc(shard), clientsCashbackPrograms)

                    val clientProgramsHistory = clientPrograms.stream()
                            .map { toClientsCashbackHistoryRecord(it) }
                            ?.toList()

                    //Пишем в базу историю добавления и получаем обратно в т.ч. id записей
                    clientProgramsHistory?.let { writeClientProgramsHistory(dsl.ppc(shard), it) }
                }
    }

    private fun writeClientPrograms(dslContext: DSLContext,
                                    clientsCashbackPrograms: List<ClientsCashbackProgramsRecord>)
            : List<ClientsCashbackProgramsRecord> {

        //Получаем id'шники для связок клиентов и программ
        val ids = shardHelper.generateClientsCashbackProgramsIds(clientsCashbackPrograms.size)
        StreamEx.of(clientsCashbackPrograms)
                .zipWith(ids.stream())
                .forKeyValue { t, u -> t.clientCashbackProgramId = u }

        val helper = InsertHelper(dslContext, CLIENTS_CASHBACK_PROGRAMS)

        clientsCashbackPrograms.forEach {
            helper.set(CLIENTS_CASHBACK_PROGRAMS.CLIENT_CASHBACK_PROGRAM_ID, it.clientCashbackProgramId)
                    .set(CLIENTS_CASHBACK_PROGRAMS.CASHBACK_PROGRAM_ID, it.cashbackProgramId)
                    .set(CLIENTS_CASHBACK_PROGRAMS.CLIENT_ID, it.clientid)
                    .set(CLIENTS_CASHBACK_PROGRAMS.IS_ENABLED, it.isEnabled)
                    .set(CLIENTS_CASHBACK_PROGRAMS.LAST_CHANGE, it.lastChange)
                    .newRecord()
        }

        helper.execute()

        return clientsCashbackPrograms
    }

    private fun writeClientProgramsHistory(dslContext: DSLContext,
                                           clientsCashbackProgramsHistory: List<ClientsCashbackHistoryRecord>) {

        //Получаем id'шники для истории клиентов-программ
        val ids = shardHelper.generateClientCashbacksHistoryIds(clientsCashbackProgramsHistory.size)
        StreamEx.of(clientsCashbackProgramsHistory)
                .zipWith(ids.stream())
                .forKeyValue { t, u -> t.clientCashbackHistoryId = u }

        val helper = InsertHelper(dslContext, CLIENTS_CASHBACK_HISTORY)

        clientsCashbackProgramsHistory.forEach {
            helper.set(CLIENTS_CASHBACK_HISTORY.CLIENT_CASHBACK_HISTORY_ID, it.clientCashbackHistoryId)
                    .set(CLIENTS_CASHBACK_HISTORY.CLIENT_CASHBACK_PROGRAM_ID, it.clientCashbackProgramId)
                    .set(CLIENTS_CASHBACK_HISTORY.STATE_CHANGE, it.stateChange)
                    .set(CLIENTS_CASHBACK_HISTORY.CHANGE_TIME, it.changeTime)
                    .newRecord()
        }

        helper.execute()
    }

    private fun getClientsForCashback(): List<CashbackClientInfo> {
        val cluster = YtCluster.HAHN
        val result: MutableList<CashbackClientInfo> = ArrayList()
        //Достаем все из YT, фильтруем по полям трекера и конвертируем в удобный формат
        ytProvider.getOperator(cluster).readTable(ytCashbackClients,
                YtCashbackClientsRow(listOf(
                        YtCashbackClients.ID,
                        YtCashbackClients.COMPONENTS,
                        YtCashbackClients.CREATED,
                        YtCashbackClients.CUSTOMFIELDS,
                        YtCashbackClients.STATUS)),
                { ytCashbackClientsRow: YtCashbackClientsRow -> convertAndFilterFromYt(ytCashbackClientsRow, result) })
        return result
    }

    private fun convertAndFilterFromYt(row: YtCashbackClientsRow, result: MutableList<CashbackClientInfo>) {
        val components = row.components.asList().stream()
                .map { obj: YTreeNode -> obj.stringValue() }
                .collect(Collectors.toList())
        val customFields = row.customFields.asMap()

        //Все условия написаны на основе скрипта, с помощью которого сейчас вычисляются клиенты участвующие в программах
        if (customFields.containsKey(customFieldClientIdName)
                && customFields.containsKey(customFieldConferenceName)
                && customFields.get(customFieldConferenceName)?.stringValue() == "RUB"
                && components.containsAll(listOf(componentsGotInitMail, componentsGoodLogin))
                && !components.contains(componentsNotDuplicate)
                && !components.contains(componentsNotOnHold)) {

            result.add(CashbackClientInfo(
                    clientId = java.lang.Long.valueOf(customFields.get(customFieldClientIdName)?.stringValue()),
                    archived = row.status == statusArchived,
                    added = LocalDateTime.ofInstant(Instant.ofEpochMilli(row.created),
                            MoscowTime.TZ.toTimeZone().toZoneId()))
            )
        }
    }

    private fun toClientsCashbackHistoryRecord(client: ClientsCashbackProgramsRecord)
            : ClientsCashbackHistoryRecord = ClientsCashbackHistoryRecord(
            0L,
            client.clientCashbackProgramId,
            if (client.isEnabled == 1L) {
                ClientsCashbackHistoryStateChange.`in`
            } else {
                ClientsCashbackHistoryStateChange.out
            },
            client.lastChange
    )

    private fun toClientsCashbackProgramsRecord(client: CashbackClientInfo, programId: Long, enabled: Long)
            : ClientsCashbackProgramsRecord = ClientsCashbackProgramsRecord(
            0L,
            client.clientId,
            programId,
            enabled,
            client.added
    )

}
