package ru.yandex.direct.oneshot.oneshots.importcashbackdata

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.LAST_IMPORTED_CASHBACK_MONTH
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.core.entity.cashback.model.CashbackRewardsImportParams
import ru.yandex.direct.core.entity.cashback.service.ImportCashbackRewardsService
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.StringConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.YtPathUtil
import ru.yandex.direct.ytwrapper.model.YtCluster
import java.time.LocalDate
import java.time.format.DateTimeFormatter
import java.util.regex.Pattern

data class CashbackImportDetailsParam(
        val clusterName: String,
        val tableName: String,
        val updateLastImportedProperty: Boolean
)

/**
 * Oneshot для многократного применения.
 * Переносит данные о кешбеках начисленных клиентам
 * В качестве параметра получает кластер, имя таблицы в дефолтной папке
 * Обычно это автоматическое действие, которым занимается Job с именем ImportCashbackRewardsDetailsJob,
 * но иногда случаются внезапные нештатные выгрузки. Если джоба не смогла, потому что правильная таблица
 * получила постфикс или есть дополнительная выгрузка по новым клиентам/акциям (клиент+акция+месяц — уникальный ключ),
 * то можно воспользоваться этим oneshot'ом.
 * Если таким образом импортируется основная(!) таблица с детализацией, то стоит проставить параметр
 * updateLastImportedProperty в true, тогда джоба будет считать, что импорт в этом месяце был и не будет расстраиваться
 */
@Component
@Multilaunch
@Approvers("kozobrodov", "pavryabov", "buhter", "mivolgin")
class ImportCashbackDetailsRegularOneshot @Autowired constructor(
        private val rewardsService: ImportCashbackRewardsService,
        private val ppcPropertiesSupport: PpcPropertiesSupport,
        private val directConfig: DirectConfig,
) : SimpleOneshot<CashbackImportDetailsParam, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(ImportCashbackDetailsRegularOneshot::class.java)
        private val dateFormatter = DateTimeFormatter.ofPattern("yyyyMM")

        //поддерживаются таблицы с названием типа "yyyyMM" или "yyyyMM_0"-"yyyyMM_99"
        private val tableNamePattern = Pattern.compile("\\p{Nd}{6}+(?:_[0-9]{1,2})?")
    }

    override fun validate(inputData: CashbackImportDetailsParam):
            ValidationResult<CashbackImportDetailsParam, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        vb.item(inputData.tableName, "tableName")
                .check(CommonConstraints.notNull())
                .check(StringConstraints.notBlank())
                .check(StringConstraints.matchPattern(tableNamePattern))
        vb.item(inputData.clusterName, "cluster")
                .check(CommonConstraints.notNull())
                .check(StringConstraints.notBlank())
                .check(CommonConstraints.inSet(YtCluster.getNames()))

        return vb.result
    }

    override fun execute(inputData: CashbackImportDetailsParam, prevState: Void?): Void? {
        val cfg: DirectConfig = directConfig.getBranch("cashbacks.yt")
        val baseYtPath = cfg.getString("rewards_table_base_path")
        val path = YtPathUtil.generatePath(baseYtPath, inputData.tableName)
        logger.info("Start processing ${path.toString()} table")
        val date = LocalDate.of(inputData.tableName.subSequence(0, 4).toString().toInt(),
                inputData.tableName.subSequence(4, 6).toString().toInt(),
                1)
        logger.info("Parsed month: ${dateFormatter.format(date)}")
        rewardsService.importRewardsTable(CashbackRewardsImportParams()
                .withCluster(YtCluster.parse(inputData.clusterName))
                .withDate(date)
                .withTablePath(path))

        if (inputData.updateLastImportedProperty) {
            logger.info("Updating property ${LAST_IMPORTED_CASHBACK_MONTH.name} to parsed month")
            ppcPropertiesSupport.get(LAST_IMPORTED_CASHBACK_MONTH).set(date)
            logger.info("Updated property ${LAST_IMPORTED_CASHBACK_MONTH.name} to parsed month")
        }
        logger.info("Finish processing ${inputData.tableName} table")
        return null
    }

}
