package ru.yandex.direct.oneshot.oneshots.interstitialtoinapp;

import java.util.List;
import java.util.Set;

import com.fasterxml.jackson.core.type.TypeReference;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.InventoryType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbschema.ppcdict.tables.CpmPricePackages.CPM_PRICE_PACKAGES;


@Component
@Multilaunch
@PausedStatusOnFail
@Approvers({"andreypav"})
public class RenameInterstitialToInAppPricePackagesOneShot implements SimpleOneshot<Void, Void> {

    private static final Logger logger = LoggerFactory.getLogger(RenameInterstitialToInAppPricePackagesOneShot.class);

    private final DslContextProvider dslContextProvider;

    @Autowired
    public RenameInterstitialToInAppPricePackagesOneShot(
            DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    private List<PricePackageInfo> getBidModifiersToUpdate() {
        return dslContextProvider.ppcdict()
                .select(CPM_PRICE_PACKAGES.PACKAGE_ID, CPM_PRICE_PACKAGES.BID_MODIFIERS)
                .from(CPM_PRICE_PACKAGES)
                .where(CPM_PRICE_PACKAGES.BID_MODIFIERS.like("%INTERSTITIAL%"))
                .fetch(rec -> new PricePackageInfo(rec.value1(), rec.value2()));
    }
    private Integer getInAppBidModifiers() {
        return dslContextProvider.ppcdict()
                .selectCount()
                .from(CPM_PRICE_PACKAGES)
                .where(CPM_PRICE_PACKAGES.BID_MODIFIERS.like("%INAPP%"))
                .fetchOne().value1();
    }

    private void updateBidModifiers(PricePackageInfo pricePackageInfo) {
        dslContextProvider.ppcdict()
                .update(CPM_PRICE_PACKAGES)
                .set(CPM_PRICE_PACKAGES.BID_MODIFIERS,pricePackageInfo.getBidModifiers())
                .where(CPM_PRICE_PACKAGES.PACKAGE_ID.eq(pricePackageInfo.getPackageId()))
                .execute();
    }

    @Nullable
    @Override
    public Void execute(Void inputData, @Nullable Void prevState) {
        logger.info("RenameInterstitialToInAppPricePackagesOneShot START");
        Integer inAppBeforeUpdate = getInAppBidModifiers();
        List<PricePackageInfo> records = getBidModifiersToUpdate();
        records.forEach(record -> {
            Set<BidModifier> bidModifiers = JsonUtils.fromJson(record.getBidModifiers(), new TypeReference<>() {});
            bidModifiers.forEach( bidModifier -> {
                if (bidModifier instanceof BidModifierInventory) {
                    BidModifierInventory bidModifierInventory = (BidModifierInventory) bidModifier;
                    bidModifierInventory.getInventoryAdjustments().forEach(adj -> {
                        if (adj.getInventoryType() == InventoryType.INTERSTITIAL) {
                        logger.info("Update bidModifier for package " + record.getPackageId());
                            adj.setInventoryType(InventoryType.INAPP);
                        }
                    });

                }
            });
            record.setBidModifiers(JsonUtils.toJson(bidModifiers, new TypeReference<Set<BidModifier>>() {}));

            updateBidModifiers(record);

        });
        Integer inAppAfterUpdate = getInAppBidModifiers();
        if (records.size() + inAppBeforeUpdate == inAppAfterUpdate) {
            logger.info(String.format("Total updated: %d. Control check is success", records.size()));
        } else {
            logger.info(String.format("Total updated: %d. Control check is NOT success. Please, re-check manually.", records.size()));
        }
        logger.info("RenameInterstitialToInAppPricePackagesOneShot FINISH");

        return null;
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }

    private static class PricePackageInfo {
        final Long packageId;
        String bidModifiers;

        public PricePackageInfo(Long packageId, String bidModifiers) {
            this.packageId = packageId;
            this.bidModifiers = bidModifiers;
        }

        public Long getPackageId() {
            return packageId;
        }

        public String getBidModifiers() {
            return bidModifiers;
        }

        public void setBidModifiers(String bidModifiers) {
            this.bidModifiers = bidModifiers;
        }
    }
}
