package ru.yandex.direct.oneshot.oneshots.invalidpermalinks;

import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.ytwrapper.client.YtProvider;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Ваншот для распубликации пермалинков по таблице из YTя.
 * Берет shard, ClientID и permalink_id из таблицы в YT, в таблице organizations по ключу {ClientID, permalink_id} в
 * шарде shard меняет статус публикации на UNKNOWN, отправляет на перемодерацию и переотправку в БК банеры, к которым
 * прилинкованы эти организации
 * https://st.yandex-team.ru/DIRECT-124583
 * <p>
 * Ваншот ожидает два обязательных входных параметра - строку с кластером YT, на котором лежит таблица
 * и строку с путем к таблице с данными
 */
@Component
@Approvers({"ppalex", "hmepas", "lena-san", "gerdler", "maxlog", "zhur", "kuhtich", "hrustyashko", "pavryabov"})
@Multilaunch
public class UnpublishInvalidPermalinksOneshot extends InvalidPermalinksOneshot {
    private static final Logger logger = LoggerFactory.getLogger(UnpublishInvalidPermalinksOneshot.class);
    private static final long CHUNK_SIZE = 100;

    private final DslContextProvider dslContextProvider;
    private final OrganizationRepository organizationRepository;
    private final BannerModerationRepository bannerModerationRepository;

    public UnpublishInvalidPermalinksOneshot(
            YtProvider ytProvider,
            DslContextProvider dslContextProvider,
            OrganizationRepository organizationRepository,
            BannerModerationRepository bannerModerationRepository
    ) {
        super(ytProvider, CHUNK_SIZE);
        this.dslContextProvider = dslContextProvider;
        this.organizationRepository = organizationRepository;
        this.bannerModerationRepository = bannerModerationRepository;
    }

    @Override
    protected void processItems(int shard, List<Pair<ClientId, Long>> items) {
        if (items.isEmpty()) {
            return;
        }
        List<Long> linkedBannerIds =
                organizationRepository.getLinkedBannerIdsByClientIdPermalinkIds(shard, items);

        dslContextProvider.ppc(shard).transaction(ctx -> {
            DSLContext context = DSL.using(ctx);

            if (!linkedBannerIds.isEmpty()) {
                bannerModerationRepository.updateStatusModerate(context, linkedBannerIds,
                        BannerStatusModerate.READY);
            }
            organizationRepository.updateOrganizationsStatusPublishByPermalinkIds(context,
                    mapList(items, Pair::getRight), OrganizationStatusPublish.UNKNOWN);
        });
    }

    @Override
    protected Logger getLogger() {
        return logger;
    }
}
