package ru.yandex.direct.oneshot.oneshots.json_settings.feedback_sessions_update

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.grid.core.frontdb.jsonsettings.service.JsonSettingsService
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.time.LocalDate

data class Param(
        val logins: List<String>
)

/**
 * Многоразовый ваншот.
 * Для выбранных логинов в JsonSettings в YDB проставляет измененные значения сессий, таким образом,
 * чтобы вызвать принудительный показ ФОС
 * Ограничено 100 логинами за раз
 */
@Component
@Multilaunch
@Approvers("alkaline", "collapsus", "evolkowa", "buhter")
class FeedbackSessionsUpdateOneshot @Autowired constructor(
        private val jsonSettingsService: JsonSettingsService,
        private val userService: UserService
) : SimpleOneshot<Param, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(
                FeedbackSessionsUpdateOneshot::class.java
        )
    }

    /**
     * Проверяем, что передан непустой список логинов
     */
    override fun validate(inputData: Param): ValidationResult<Param, Defect<*>> {
        return ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.logins, "logins")
                    .check(CommonConstraints.notNull())
                    .check(CollectionConstraints.notEmptyCollection())
                    .check(CollectionConstraints.maxListSize(100))
        }.result
    }

    override fun execute(inputData: Param, prevState: Void?): Void? {
        logger.info("got {} logins to process", inputData.logins.size)
        val uids = userService.massGetUidByLogin(inputData.logins).values.distinct()
        logger.info("found {} uids by logins", uids.size)
        for (uid in uids) {
            jsonSettingsService.feedbackWriteSessionsDates(uid, LocalDate.now().minusDays(2), LocalDate.now().minusDays(1))
        }
        return null
    }

}
