package ru.yandex.direct.oneshot.oneshots.logs;

import java.time.Instant;
import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.util.List;
import java.util.stream.Collectors;

import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.oneshot.base.SimpleYtOneshot;
import ru.yandex.direct.oneshot.base.YtInputData;
import ru.yandex.direct.oneshot.base.YtState;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.ListNode;
import ru.yandex.yt.ytclient.proxy.request.SetNode;

import static java.time.format.DateTimeFormatter.ISO_LOCAL_DATE;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.ytwrapper.YtUtils.EXPIRATION_TIME_ATTR;

/**
 * Ставит трехлетний expiration_time на основании даты таблицы. Принимает кластер и папку.
 */
@Multilaunch
@Component
@Approvers({"ppalex", "maxlog", "zhur", "pavryabov"})
public class SetLogsExpirationTime extends SimpleYtOneshot<YtInputData, YtState> {
    private static final Logger logger = LoggerFactory.getLogger(SetLogsExpirationTime.class);

    @Autowired
    public SetLogsExpirationTime(YtProvider ytProvider) {
        super(ytProvider);
    }

    @Nullable
    @Override
    public YtState execute(YtInputData inputData, @Nullable YtState prevState) {
        YtCluster cluster = YtCluster.parse(inputData.getYtCluster());
        String pathPrefix = inputData.getTablePath();
        logger.info("Processing {} on cluster {}", pathPrefix, cluster);

        YtDynamicOperator operator = ytProvider.getDynamicOperator(cluster);

        logger.info("fetch tables list");
        var listReq = new ListNode(pathPrefix).setAttributes(ColumnFilter.of(EXPIRATION_TIME_ATTR));
        List<YTreeNode> allTables = operator.runRpcCommandWithTimeout(operator.getYtClient()::listNode, listReq)
                .asList();

        logger.info("got {} tables", allTables.size());

        List<String> tablesWithoutAttribute = allTables.stream()
                .filter(n -> !n.getAttributes().containsKey(EXPIRATION_TIME_ATTR))
                .map(YTreeNode::stringValue)
                .filter(this::isValidDate)
                .collect(Collectors.toList());
        logger.info("Found {} YYYY-MM-DD named tables without expiration time", tablesWithoutAttribute.size());

        for (String date : tablesWithoutAttribute) {
            YPath attribute = YPath.simple(pathPrefix).child(date).attribute(EXPIRATION_TIME_ATTR);
            Instant expiration = calcExpirationTime(date);
            logger.info("Setting {} to {}", attribute, expiration);

            var setReq = new SetNode(attribute, YTree.longNode(expiration.toEpochMilli()));
            operator.runRpcCommandWithTimeout(operator.getYtClient()::setNode, setReq);
        }

        logger.info("all tables processed");

        return null;
    }

    private static LocalDate parse(String date) {
        return LocalDate.parse(date, ISO_LOCAL_DATE);
    }

    private static Instant calcExpirationTime(String date) {
        return parse(date)
                .atStartOfDay()     // переводим в LocalDateTime
                .plusDays(1)        // берем начало следующих суток
                .minusSeconds(1)    // берем конец предыдущих суток
                .plusYears(3)       // прибавляем желаемый TTL
                .atZone(MSK)
                .toInstant();
    }


    private boolean isValidDate(String name) {
        try {
            parse(name);
            return true;
        } catch (DateTimeParseException e) {
            return false;
        }
    }
}
