package ru.yandex.direct.oneshot.oneshots.market_campaigns_allowed_pageids

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.placements.repository.PlacementsRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.utils.FunctionalUtils.mapList
import ru.yandex.direct.utils.ThreadUtils
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.listProperty

data class InputData(
    val clientIds: List<Long>,
    val campaignIds: List<Long>,
    val allowedZenPageIds: List<Long>,
    val disallowedZenPageIds: List<Long>,
)

/**
 * Таргетирует переданные кампании Маркета на указанные пейджи Дзена в ПП,
 * проставляя allowedZenPageIds в `allowed_page_ids` в `camp_options`.
 * Потом запрещает показ на `disallowedZenPageIds` для остальных кампаний переданных клиентов, которые имеют тип Смарт-Баннер
 * и не входят в список входных campaignIds, проставляя disallowedZenPageIds в `disallowed_page_ids` в `camp_options`.
 */
@Component
@Approvers("bratgrim", "pavelkataykin")
@Multilaunch
class SetAllowedPageIdsForMarketCampaignsOneshot @Autowired constructor(
    private val placementsRepository: PlacementsRepository,
    private val campaignRepository: CampaignRepository,
    private val bsResyncService: BsResyncService,
    private val shardHelper: ShardHelper
) : SimpleOneshot<InputData, Void> {

    companion object {
        private val logger = LoggerFactory.getLogger(SetAllowedPageIdsForMarketCampaignsOneshot::class.java)
        private const val CHUNK_SIZE: Int = 200
        private const val RELAX_TIME: Long = 3_000L
    }

    override fun validate(inputData: InputData): ValidationResult<InputData, Defect<*>> {
        return ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            listProperty(inputData::campaignIds)
                .check(notNull())
                .check(CollectionConstraints.notEmptyCollection())
                .checkEach(validId(), When.isValid())
            listProperty(inputData::clientIds)
                .check(notNull())
                .check(CollectionConstraints.notEmptyCollection())
                .checkEach(validId(), When.isValid())
                .checkEach(
                    Constraint.fromPredicate(
                        { clientId -> shardHelper.isExistentClientId(clientId!!) }, CommonDefects.objectNotFound()
                    ), When.isValid()
                )
            listProperty(inputData::allowedZenPageIds)
                .check(notNull())
                .check(placementsExist(), When.isValid())
            listProperty(inputData::disallowedZenPageIds)
                .check(notNull())
                .check(placementsExist(), When.isValid())
        }.result
    }

    override fun execute(inputData: InputData, prevState: Void?): Void? {
        setAllowedPagesForCampaigns(inputData)
        setDisallowedPageIdsForSmartCampaigns(inputData)
        return null
    }

    private fun setAllowedPagesForCampaigns(inputData: InputData) {
        //получаем по шардам существующие id кампаний и обновляем для них allowed_page_ids
        logger.info("Setting allowed pages ${inputData.allowedZenPageIds}")
        val shardToCampaign = shardHelper.groupByShard(inputData.campaignIds, ShardKey.CID)
        shardToCampaign.forEach { shard, campaignIds ->
            logger.info("Processing shard $shard, setting allowed pages for cids $campaignIds")
            val cidToAllowedPages = campaignRepository.getCampaignsAllowedPageIds(shard, campaignIds)
            campaignIds
                .forEach { cid ->
                    val allowedPageIds = cidToAllowedPages.getOrDefault(cid, emptyList()).toMutableList()
                    logger.info("Shard $shard, campaign $cid with allowed pages: $allowedPageIds")
                    allowedPageIds.addAll(inputData.allowedZenPageIds)
                    campaignRepository
                        .updateAllowedPageIds(shard, cid, allowedPageIds)
                    logger.info("Shard $shard, campaign $cid with new allowed pages: $allowedPageIds")
                }
            resyncWithBs(shard, campaignIds)
        }
    }

    private fun setDisallowedPageIdsForSmartCampaigns(inputData: InputData) {
        val campaignIds = inputData.campaignIds

        logger.info("Setting disallowed pages $inputData.disallowedZenPageIds")
        val clientIdsByShard = shardHelper.groupByShard(inputData.clientIds, ShardKey.CLIENT_ID)
        clientIdsByShard.forEach { shard, clientIds ->
            val updatedCampaignIds = mutableListOf<Long>()
            //получаем кампании типа смарт-баннеры, привязанные к входным ClientId
            val smartCampaignsByShard = campaignRepository
                .getNotEmptySmartCampaignsByClientIds(shard, mapList(clientIds, ClientId::fromLong))
                .chunked(CHUNK_SIZE)
            for (campaignsChunk in smartCampaignsByShard) {
                val cidToDisallowedPages = campaignRepository.getCampaignsDisallowedPageIds(shard, campaignsChunk)
                campaignsChunk.forEach { campaignId ->
                    //если кампания не входит в список обработанных в setAllowedPagesForCampaigns, обновляем для нее disallowed_page_ids
                    if (!campaignIds.contains(campaignId)) {
                        val disallowedPageIds =
                            cidToDisallowedPages.getOrDefault(campaignId, emptyList()).toMutableList()
                        logger.info("Shard $shard, campaign $campaignId with disallowed pages: $disallowedPageIds")
                        disallowedPageIds.addAll(inputData.disallowedZenPageIds)
                        campaignRepository
                            .updateDisallowedPageIds(shard, campaignId, disallowedPageIds)
                        logger.info("Shard $shard, new disallowed pages $disallowedPageIds for campaign $campaignId")
                        updatedCampaignIds.add(campaignId)
                    }
                }
                ThreadUtils.sleep(RELAX_TIME)
            }
            resyncWithBs(shard, updatedCampaignIds)
        }
    }

    /**
     * Переотправка обновленных данных кампаний в БК
     */
    fun resyncWithBs(shard: Int, campaignIds: Collection<Long>) {
        for (campaignsChunk in campaignIds.chunked(CHUNK_SIZE)) {
            val bsResyncItems = campaignsChunk
                .map { BsResyncItem(BsResyncPriority.ON_CHANGED_RETARGETING_ACCESSIBILITY, it) }
            val resyncDataAmount = bsResyncService.addObjectsToResync(bsResyncItems)
            logger.info("Shard $shard, added $resyncDataAmount campaigns into bs_resync_queue table")
        }
    }

    fun placementsExist(): Constraint<List<Long>, Defect<*>> {
        return Constraint { pageIds: List<Long> ->
            val existingPageIds = placementsRepository.getExistingPageIds(pageIds)
            if (existingPageIds.containsAll(pageIds)) null else CommonDefects.objectNotFound()
        }
    }
}
