package ru.yandex.direct.oneshot.oneshots.marketfeeds

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.oneshot.oneshots.marketfeeds.repository.AdGroupWithFeedIdRepository
import ru.yandex.direct.oneshot.oneshots.marketfeeds.repository.AdGroupWithFeedIdRepository.Companion.feedsByFeedIdsCondition
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateObject

data class ReplaceFeedInfo(
    val old: Long,
    val new: Long
)

data class FeedsToReplaceInfo(
    val feedIds: List<ReplaceFeedInfo>
)

private val LOGGER = LoggerFactory.getLogger(ReplaceFeedIdsInAdGroupOneshot::class.java)

private const val MAX_FEED_IDS_SIZE = 5000

/**
 * Oneshot на обновление фидов в группах (Смарты/ДО).
 *
 * Принимает список новых id фидов (`new`), на которые нужно заменить старый id (`old`).
 * При подмене фидов в группах - у них должен быть одинаковый ClientId
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov")
class ReplaceFeedIdsInAdGroupOneshot(
    private val adGroupWithFeedIdRepository: AdGroupWithFeedIdRepository,
    private val feedRepository: FeedRepository,
) : ShardedOneshot<FeedsToReplaceInfo, Void?> {

    override fun validate(inputData: FeedsToReplaceInfo): ValidationResult<FeedsToReplaceInfo, Defect<*>> {
        return validateObject(inputData) {
            listProperty(FeedsToReplaceInfo::feedIds) {
                check(notNull())
                check(notEmptyCollection(), When.isValid())
                check(maxListSize(MAX_FEED_IDS_SIZE), When.isValid())

                checkEach(unique(Comparator.comparing { it.old }), duplicatedObject(), When.isValid())
            }
        }
    }

    override fun execute(inputData: FeedsToReplaceInfo, prevState: Void?, shard: Int): Void? {
        LOGGER.info("START")

        val allFeedIds = mutableSetOf<Long>()
        allFeedIds.addAll(inputData.feedIds.map { it.new })
        allFeedIds.addAll(inputData.feedIds.map { it.old })

        val feedsByFeedIds = feedRepository.get(shard, allFeedIds)
            .associateBy { it.id }
        LOGGER.info("Found ${feedsByFeedIds.size} feeds in db")

        val feedIdsToNewFeedIds = inputData.feedIds
            .filter {
                if (!feedsByFeedIds.containsKey(it.new)) {
                    LOGGER.error("Feed with id = ${it.new} not found")
                    false
                } else if (!feedsByFeedIds.containsKey(it.old)) {
                    LOGGER.error("Feed with id = ${it.old} not found")
                    false
                } else true
            }
            .filter {
                if (feedsByFeedIds[it.new]!!.clientId != feedsByFeedIds[it.old]!!.clientId) {
                    LOGGER.error("Feeds with ids ${it.new} and ${it.old} has different clients")
                    false
                } else true
            }
            .associate { it.old to it.new }

        val condition = feedsByFeedIdsCondition(feedIdsToNewFeedIds.keys)

        val adgroupPerformanceIdsToFeedIds: Map<Long, FeedId> =
            adGroupWithFeedIdRepository.getAdGroupPerformanceIdsToFeedIds(shard, condition)
        LOGGER.info("Found ${adgroupPerformanceIdsToFeedIds.size} performance adGroups with feeds to replace")
        adGroupWithFeedIdRepository.updateAdGroupPerformanceFeedIds(
            shard, adgroupPerformanceIdsToFeedIds, feedIdsToNewFeedIds
        )

        val adgroupDynamicIdsToFeedIds: Map<Long, FeedId> =
            adGroupWithFeedIdRepository.getAdGroupDynamicIdsToFeedIds(shard, condition)
        LOGGER.info("Found ${adgroupDynamicIdsToFeedIds.size} dynamic adGroups with feeds to replace")
        adGroupWithFeedIdRepository.updateAdGroupDynamicFeedIds(
            shard, adgroupDynamicIdsToFeedIds, feedIdsToNewFeedIds
        )

        LOGGER.info("FINISH")
        return null
    }
}
