package ru.yandex.direct.oneshot.oneshots.marketfeeds

import java.util.HashSet
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.feed.DEFAULT_REFRESH_INTERVAL
import ru.yandex.direct.core.entity.feed.createFakeFeedUrl
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedRow
import ru.yandex.direct.core.entity.feed.model.UpdateStatus
import ru.yandex.direct.core.entity.feed.repository.DataCampFeedYtRepository
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.oneshots.marketfeeds.repository.AdGroupWithFeedIdRepository
import ru.yandex.direct.oneshot.oneshots.marketfeeds.repository.AdGroupWithFeedIdRepository.Companion.marketFeedsByClientIdsCondition
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.constraint.CommonConstraints.isNull
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

typealias ShopId = Long
typealias FeedId = Long

data class ClientsForUpdatingFeeds(
    val clientIds: List<Long>?,
    val updateAllClients: Boolean
)

private const val MAX_CLIENTS_SIZE = 5000
private const val CLIENTS_CHUNK_SIZE = 1000
private const val FEED_NAME_BY_SHOP_NAME_PATTERN = "Фид из Яндекс.Маркета по магазину %s"

private val LOGGER = LoggerFactory.getLogger(UpdateMarketFeedsByShopForAdGroupsOneshot::class.java)

/**
 * Oneshot на обновление маркетных фидов в группах (Смарты/ДО) по одному магазину.
 *
 * Для переданных клиентов в `clientIds`, или по всем клиентам, у которых есть маркетные фиды (`updateAllClients == true`)
 * ищутся фиды, используемые в группах, у которых `master_system == market`.
 * Для всех найденных фидов выбирается только один фид на магазин (по `marketShopId`).
 * В соответствии с магазинами берется выгрузка из YT маркетных фидов (также один фид на магазин).
 * Далее id фидов в группах обновляются в соответствии с `shopId` ранее используемого фида.
 * Фиды обновляются на основе выгрузки из yt, при этом `marketFeedId = marketShopId`.
 * Более неиспользуемые фиды будут удалены в джобе SyncMarketFeedsJob`.
 *
 * После выполнения ваншота для клиентов нужно установить фичу `SYNC_MARKET_FEEDS_BY_SHOP_ID`,
 * что бы обновленные фиды не были удалены в джобе `SyncMarketFeedsJob`.
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov")
class UpdateMarketFeedsByShopForAdGroupsOneshot(
    private val adGroupWithFeedIdRepository: AdGroupWithFeedIdRepository,
    private val feedRepository: FeedRepository,
    private val feedService: FeedService,
    private val dataCampFeedYtRepository: DataCampFeedYtRepository,
    private val shardSupport: ShardSupport,
    private val clientRepository: ClientRepository
) : ShardedOneshot<ClientsForUpdatingFeeds, Void?> {
    override fun validate(inputData: ClientsForUpdatingFeeds): ValidationResult<ClientsForUpdatingFeeds, Defect<*>> {
        return validateObject(inputData) {
            property(ClientsForUpdatingFeeds::clientIds) {
                if (inputData.updateAllClients) {
                    check(isNull())
                } else {
                    check(notNull())
                    check(notEmptyCollection(), When.isValid())
                    check(maxListSize(MAX_CLIENTS_SIZE), When.isValid())
                }
            }
        }
    }

    override fun execute(inputData: ClientsForUpdatingFeeds, prevState: Void?, shard: Int): Void? {
        LOGGER.info("START")

        val clients = if (inputData.updateAllClients) {
            LOGGER.info("Find all clients with market feeds")
            getClientsWithMarketFeeds(shard)
        } else {
            LOGGER.info("Find clients from input data")
            getClientsByClientIds(shard, inputData.clientIds ?: emptyList())
        }

        val dataCampFeeds = if (clients.isNotEmpty()) {
            LOGGER.info("Found ${clients.size} clients")

            val clientIds = clients
                .mapTo(HashSet()) { it.clientId }
            dataCampFeedYtRepository.getYtFeedsRows(clientIds)
        } else {
            LOGGER.info("Clients not found")
            emptyList()
        }

        clients.chunked(CLIENTS_CHUNK_SIZE) {
            val clientsByClientId = clients
                .associateBy { it.clientId }
            processClients(shard, clientsByClientId, dataCampFeeds)
        }

        LOGGER.info("FINISH")
        return null
    }

    private fun processClients(shard: Int, clients: Map<Long, Client>, dataCampFeeds: List<FeedRow>) {
        val clientIds = clients.keys

        val condition = marketFeedsByClientIdsCondition(clientIds)
        val feedIdsByAdGroupPerformance: Map<Long, FeedId> =
            adGroupWithFeedIdRepository.getAdGroupPerformanceIdsToFeedIds(shard, condition)
        val feedIdsByAdGroupDynamic: Map<Long, FeedId> =
            adGroupWithFeedIdRepository.getAdGroupDynamicIdsToFeedIds(shard, condition)

        val feedIds = mutableSetOf<FeedId>()
        feedIds.addAll(feedIdsByAdGroupPerformance.values)
        feedIds.addAll(feedIdsByAdGroupDynamic.values)

        if (feedIds.isEmpty()) {
            LOGGER.info("AdGroups with used feeds not found")
            return
        }

        val dbFeeds = feedRepository.get(shard, feedIds)
        val shopIdsToFeeds: Map<ShopId, Feed> = dbFeeds
            .filter { it.updateStatus != UpdateStatus.ERROR }
            .groupBy { it.marketShopId }
            .mapValues { feeds -> feeds.value.minByOrNull { it.marketFeedId }!! }

        LOGGER.info("Found ${shopIdsToFeeds.size} used shops")

        val filteredDataCampFeeds = dataCampFeeds
            .filter { shopIdsToFeeds.contains(it.shopId) }

        if (filteredDataCampFeeds.isEmpty()) {
            LOGGER.info("Shop ids not found in YT")
            return
        }

        val dataCampShopIds = filteredDataCampFeeds.map { it.shopId }
            .toSet()

        LOGGER.info("Found ${dataCampShopIds.size} shops from yt")

        val feedIdsToNewFeedIds = dbFeeds
            .associate { it.id to it.marketShopId }
            .filterValues { shopId ->
                if (!dataCampShopIds.contains(shopId)) {
                    LOGGER.info("Feed with shopId = $shopId not found in YT")
                    false
                } else true
            }
            .mapValues { shopIdsToFeeds[it.value]!!.id }

        adGroupWithFeedIdRepository.updateAdGroupPerformanceFeedIds(
            shard,
            feedIdsByAdGroupPerformance,
            feedIdsToNewFeedIds
        )
        adGroupWithFeedIdRepository.updateAdGroupDynamicFeedIds(shard, feedIdsByAdGroupDynamic, feedIdsToNewFeedIds)

        val shopIdsToModelChanges = dataCampFeedsToModelChanges(filteredDataCampFeeds, shopIdsToFeeds)
        updateFeeds(clients, shopIdsToFeeds, shopIdsToModelChanges)
    }

    private fun getClientsByClientIds(shard: Int, clientIds: List<Long>): List<Client> {
        val shards = shardSupport.getShards(ShardKey.CLIENT_ID, clientIds)
        val filteredClientIds = clientIds.asSequence()
            .zip(shards.asSequence())
            .filter { it.second == shard }
            .mapTo(HashSet()) { ClientId.fromLong(it.first) }

        return clientRepository.get(shard, filteredClientIds)
    }

    private fun getClientsWithMarketFeeds(shard: Int): List<Client> {
        val clientIds = adGroupWithFeedIdRepository.getClientIdsWithMarketFeeds(shard)
            .mapTo(HashSet()) { ClientId.fromLong(it) }

        return clientRepository.get(shard, clientIds)
    }

    private fun dataCampFeedsToModelChanges(
        dataCampFeeds: Collection<FeedRow>,
        dbFeedsByShopId: Map<ShopId, Feed>
    ): Map<ShopId, ModelChanges<Feed>> =
        dataCampFeeds.groupBy { it.shopId }
            .mapValues { it.value.minByOrNull(FeedRow::feedId)!! }
            .mapValues { (_, dataCampFeed) ->
                val dbFeed = dbFeedsByShopId[dataCampFeed.shopId]!!

                ModelChanges(dbFeed.id, Feed::class.java).apply {
                    process(dataCampFeed.businessId, Feed.MARKET_BUSINESS_ID)
                    process(dataCampFeed.shopId, Feed.MARKET_SHOP_ID)
                    process(dataCampFeed.shopId, Feed.MARKET_FEED_ID)

                    val name = String.format(FEED_NAME_BY_SHOP_NAME_PATTERN, dataCampFeed.shopName)
                    process(name, Feed.NAME)

                    val url =
                        createFakeFeedUrl(
                            dataCampFeed.businessId,
                            dataCampFeed.shopId,
                            null,
                            dataCampFeed.url,
                            true,
                            true
                        )
                    process(url, Feed.URL)

                    process(dataCampFeed.login, Feed.LOGIN)
                    process(dataCampFeed.password, Feed.PLAIN_PASSWORD)
                    process(UpdateStatus.UPDATING, Feed.UPDATE_STATUS)
                    process(DEFAULT_REFRESH_INTERVAL, Feed.REFRESH_INTERVAL)
                    process(dataCampFeed.shopName, Feed.SHOP_NAME)
                }
            }

    private fun updateFeeds(
        clients: Map<Long, Client>,
        shopIdsToFeed: Map<ShopId, Feed>,
        shopIdsToModelChanges: Map<ShopId, ModelChanges<Feed>>
    ) {
        shopIdsToModelChanges.forEach { (shopId, modelChanges) ->
            val feed = shopIdsToFeed[shopId]!!
            val clientId = feed.clientId
            val client = clients[clientId]!!
            val operatorUid = client.agencyUserId ?: client.chiefUid
            val uidAndClientId = UidAndClientId.of(client.chiefUid, ClientId.fromLong(client.id))
            LOGGER.info("Updating feed with shopId = $shopId")
            updateFeed(feed.id, operatorUid, uidAndClientId, modelChanges)
        }
    }

    private fun updateFeed(
        feedId: FeedId,
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        modelChanges: ModelChanges<Feed>
    ): FeedId? {
        val massResult =
            feedService.updateFeeds(uidAndClientId.clientId, uidAndClientId.uid, operatorUid, listOf(modelChanges))
        massResult.errors?.forEach {
            LOGGER.error(
                "Error when update a market feed: feedId: $feedId, clientId:{${uidAndClientId.clientId}}, defect: {$it}"
            )
        }
        return massResult.result
            ?.map { it.result }
            ?.firstOrNull()
    }
}
