package ru.yandex.direct.oneshot.oneshots.marketfeeds.repository

import org.jooq.Condition
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.FEEDS
import ru.yandex.direct.dbschema.ppc.enums.FeedsMasterSystem
import ru.yandex.direct.dbutil.QueryWithoutIndex
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder
import ru.yandex.direct.model.ModelChanges

@Repository
class AdGroupWithFeedIdRepository(
    private val dslContextProvider: DslContextProvider,
    private val adGroupRepository: AdGroupRepository
) {
    companion object {
        private val LOGGER = LoggerFactory.getLogger(AdGroupWithFeedIdRepository::class.java)

        fun marketFeedsByClientIdsCondition(clientIds: Collection<Long>): Condition {
            return FEEDS.CLIENT_ID.`in`(clientIds)
                .and(FEEDS.MASTER_SYSTEM.eq(FeedsMasterSystem.market))
        }

        fun feedsByFeedIdsCondition(feedIds: Collection<Long>): Condition {
            return FEEDS.FEED_ID.`in`(feedIds)
        }
    }

    @QueryWithoutIndex("Вызывается только из oneshot'a")
    fun getClientIdsWithMarketFeeds(shard: Int): List<Long> {
        return dslContextProvider.ppc(shard).selectDistinct(FEEDS.CLIENT_ID)
            .from(FEEDS)
            .where(FEEDS.MASTER_SYSTEM.eq(FeedsMasterSystem.market))
            .fetch(FEEDS.CLIENT_ID)
    }

    fun getAdGroupPerformanceIdsToFeedIds(shard: Int, condition: Condition): Map<Long, Long> {
        return dslContextProvider.ppc(shard).select(ADGROUPS_PERFORMANCE.PID, FEEDS.FEED_ID)
            .from(FEEDS)
            .join(ADGROUPS_PERFORMANCE).on(ADGROUPS_PERFORMANCE.FEED_ID.eq(FEEDS.FEED_ID))
            .where(condition)
            .fetchMap(ADGROUPS_PERFORMANCE.PID, FEEDS.FEED_ID)
    }

    fun getAdGroupDynamicIdsToFeedIds(shard: Int, condition: Condition): Map<Long, Long> {
        return dslContextProvider.ppc(shard).select(ADGROUPS_DYNAMIC.PID, FEEDS.FEED_ID)
            .from(FEEDS)
            .join(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.FEED_ID.eq(FEEDS.FEED_ID))
            .where(condition)
            .fetchMap(ADGROUPS_DYNAMIC.PID, FEEDS.FEED_ID)
    }

    fun updateAdGroupPerformanceFeedIds(
        shard: Int,
        adgroupPerformanceIdsToFeedIds: Map<Long, Long>,
        feedIdsToNewFeedIds: Map<Long, Long>
    ) {
        val adGroupIdsToAppliedChanges = adgroupPerformanceIdsToFeedIds
            .mapValues { feedIdsToNewFeedIds[it.value] }
            .filterValues { it != null }
            .mapValues {
                ModelChanges(it.key, PerformanceAdGroup::class.java)
                    .process(it.value, PerformanceAdGroup.FEED_ID)
                    .applyTo(PerformanceAdGroup().withId(it.key))
            }

        LOGGER.info("Updating ${adGroupIdsToAppliedChanges.size} performance adGroups")

        val updateBuilder = JooqUpdateBuilder(ADGROUPS_PERFORMANCE.PID, adGroupIdsToAppliedChanges.values)
        updateBuilder.processProperty(PerformanceAdGroup.FEED_ID, ADGROUPS_PERFORMANCE.FEED_ID)

        val updated = dslContextProvider.ppc(shard)
            .update(ADGROUPS_PERFORMANCE)
            .set(updateBuilder.values)
            .where(ADGROUPS_PERFORMANCE.PID.`in`(updateBuilder.changedIds))
            .execute()

        adGroupRepository.updateStatusBsSynced(shard, adGroupIdsToAppliedChanges.keys, StatusBsSynced.NO)

        LOGGER.info("Updated $updated performance adGroups")
    }

    fun updateAdGroupDynamicFeedIds(
        shard: Int,
        adgroupDynamicIdsToFeedIds: Map<Long, Long>,
        feedIdsToNewFeedIds: Map<Long, Long>
    ) {
        val adGroupIdsToAppliedChanges = adgroupDynamicIdsToFeedIds
            .mapValues { feedIdsToNewFeedIds[it.value] }
            .filterValues { it != null }
            .mapValues {
                ModelChanges(it.key, DynamicFeedAdGroup::class.java)
                    .process(it.value, DynamicFeedAdGroup.FEED_ID)
                    .applyTo(DynamicFeedAdGroup().withId(it.key))
            }

        LOGGER.info("Updating ${adGroupIdsToAppliedChanges.size} dynamic adGroups")

        val updateBuilder = JooqUpdateBuilder(ADGROUPS_DYNAMIC.PID, adGroupIdsToAppliedChanges.values)
        updateBuilder.processProperty(DynamicFeedAdGroup.FEED_ID, ADGROUPS_DYNAMIC.FEED_ID)

        val updated = dslContextProvider.ppc(shard)
            .update(ADGROUPS_DYNAMIC)
            .set(updateBuilder.values)
            .where(ADGROUPS_DYNAMIC.PID.`in`(updateBuilder.changedIds))
            .execute()

        adGroupRepository.updateStatusBsSynced(shard, adGroupIdsToAppliedChanges.keys, StatusBsSynced.NO)

        LOGGER.info("Updated $updated dynamic adGroups")
    }
}
