package ru.yandex.direct.oneshot.oneshots.marketrescuer

import com.google.common.base.Preconditions.checkState
import one.util.streamex.StreamEx
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory
import ru.yandex.direct.core.entity.banner.service.DatabaseMode
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsAddOperation
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.core.entity.vcard.service.VcardService
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.currency.CurrencyCode
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.utils.FunctionalUtils.filterList
import ru.yandex.direct.utils.FunctionalUtils.mapList
import ru.yandex.direct.utils.TextConstants.LETTERS_AND_NUMBERS
import ru.yandex.direct.utils.TextConstants.SPACE_CHARS
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeField
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeObject
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes
import java.util.Objects
import java.util.function.Consumer

data class Param(
        val ytCluster: YtCluster,
        val tablePath: String,
        val chunkSize: Int,
        val clientLogin: String,
        val metrikaCounters: List<Long>,
        val meaningfulGoalIds: List<Long>,
        val sitelinksSetId: Long,
        val calloutIds: List<Long>,
        val campaignPrefixName: String
)

data class State(
        val lastRow: Long = 0L
)

@YTreeObject
data class InputTableRow(
        @YTreeField(key = "corrected_query") val query: String
)

/**
 * Ваншот для создания кампании, групп и баннеров для эксперимента по "Спасению маркета")
 * На каждой итерации читаем из YT таблицы чанк запросов, нормализуем каждый запрос удалив невалидные символы
 * Создаем одну текстовую кампанию и визитку,
 * далее создаем для каждого запроса группу с баннером. Для группы запрос проставляем в качестве ключевой фразы,
 * а для баннера в заголовок. Для ставки проставляем максимальную для рубля - Currencies.getCurrency(CurrencyCode.RUB).maxPrice
 * Из входных параметров ваншот получает путь до YT таблицы, логин клиента для которого будет создавать объекты,
 * sitelinksSetId и calloutIds получаем, чтобы не создавать их каждый раз и использовать для всех баннеров
 * визитку к сожалению приходиться создавать, т.к. привязка только к одной кампании
 * Скрипт заточен для логина: yndx-market-rescuers
 * При смене логина, надо в корректировках поменять id условий
 * Особенности:
 * Операция по созданию групп комплексная из-за чего если хоть на одной группе или фразе сработает валидация, то остальные валидные не создадим
 * Для баннеров не так, валидные баннеры создадим, а невалидные залогируем
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers("buhter", "ssdmitriev")
class GenerateBannersForMarketOneshot @Autowired constructor(
        private val ytProvider: YtProvider,
        private val campaignOperationService: CampaignOperationService,
        private val vcardService: VcardService,
        private val userService: UserService,
        private val clientGeoService: ClientGeoService,
        private val recentStatisticsService: RecentStatisticsService,
        private val complexAdGroupAddOperationFactory: ComplexAdGroupAddOperationFactory,
        private val bannersAddOperationFactory: BannersAddOperationFactory
) : SimpleOneshot<Param, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(GenerateBannersForMarketOneshot::class.java)

        private val DECIMAL_NUMBER_WITH_COMMA_REGEX = "(\\d+)(,)(\\d+)".toRegex()
        private const val SPECIAL_CHARS = "."
        private val KEYWORD_ALLOWED_CHARS_REGEX = "[^$LETTERS_AND_NUMBERS$SPACE_CHARS$SPECIAL_CHARS]".toRegex()
        private val SPACE_SEQUENCE_REGEX = "[\\p{javaWhitespace}   ]+".toRegex()
        private val CHARS_REPLACE_TO_SPACE_REGEX = "[-/]+".toRegex()
        private val CAPITAL_LATIN_CHARS_REGEX = "[A-Z]+".toRegex()
        private val CAPITAL_CHARS_WITH_DIGITS_REGEX = "[\\p{Lu}\\d]+".toRegex()
        private val DIGIT_REGEX = "\\d".toRegex()
        private val WRONG_POINT_LEFT_REGEX = "(^|\\D)(\\.+)".toRegex()
        private val WRONG_POINT_RIGHT_REGEX = "(\\.+)(\\D|$)".toRegex()

        fun normalizeQuery(query: String): String {
            // запятую меняем на точку
            val result = DECIMAL_NUMBER_WITH_COMMA_REGEX.replace(query) { m ->
                m.groupValues[1] + "." + m.groupValues[3]
            }

            return result
                    //удаляем точку, если она не является частью числа
                    .replace(WRONG_POINT_LEFT_REGEX) { m -> m.groupValues[1] }
                    .replace(WRONG_POINT_RIGHT_REGEX) { m -> m.groupValues[2] }

                    // некоторые невалидные символы заменяем на пробел
                    .replace(CHARS_REPLACE_TO_SPACE_REGEX, " ")
                    // удаляем невалидные символы
                    .replace(KEYWORD_ALLOWED_CHARS_REGEX, "")
                    // меняем подряд идущие пробельные символы на один
                    .replace(SPACE_SEQUENCE_REGEX, " ")
                    .trim()
        }

        /**
         * Преобразовывает запрос в заголовок баннера
         * на вход ожидается текст, где удалены все лишние символы и слова разделяются только одним пробелом
         * @see normalizedQuery
         */
        fun transformQueryToTitle(normalizedQuery: String): String {
            return normalizedQuery
                    .split(" ")
                    .joinToString(separator = " ") {
                        transformWordCase(it)
                    }
        }

        /**
         * Не трогаем регистр, "если все английскими или есть цифры и все буквы большие то оставляем"
         * предлоги переводим в нижний регистр
         * в остальных случаях делаем первую букву заглавной, а остальные в нижнем регистре
         */
        private fun transformWordCase(word: String): String {
            if (CAPITAL_LATIN_CHARS_REGEX.matches(word)
                    || (CAPITAL_CHARS_WITH_DIGITS_REGEX.matches(word) && word.contains(DIGIT_REGEX))) {
                return word
            }

            if (PREPOSITIONS.contains(word)) {
                return word.toLowerCase()
            }

            return word.toLowerCase().capitalize()
        }

        fun isValidKeyword(query: String): Boolean {
            // пока достаточно этой проверки, потом может добавим еще
            val isValid = query.split(" ").size <= PhraseConstraints.WORDS_MAX_COUNT
            if (!isValid) {
                logger.warn("got invalid keyword: $query")
            }
            return isValid
        }
    }

    override fun validate(inputData: Param): ValidationResult<Param, Defect<*>>? {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)

        vb.item(inputData.ytCluster, "ytCluster")
                .check(CommonConstraints.notNull())

        if (vb.result.hasAnyErrors()) return vb.result

        vb.item(inputData.tablePath, "tablePath")
                .check(CommonConstraints.notNull())
                .check(Constraint.fromPredicate(
                        { tableName -> ytProvider.getOperator(inputData.ytCluster).exists(YtTable(tableName)) },
                        CommonDefects.objectNotFound()))

        vb.item(inputData.chunkSize, "chunkSize")
                .check(CommonConstraints.notNull())
                .check(NumberConstraints.greaterThan(0))

        vb.item(inputData.clientLogin, "clientLogin")
                .check(CommonConstraints.notNull())
        vb.item(inputData.metrikaCounters, "metrikaCounters")
                .check(CommonConstraints.notNull())
        vb.item(inputData.meaningfulGoalIds, "meaningfulGoalIds")
                .check(CommonConstraints.notNull())
        vb.item(inputData.sitelinksSetId, "sitelinksSetId")
                .check(CommonConstraints.notNull())
        vb.item(inputData.calloutIds, "calloutsIds")
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.notEmptyCollection())
                .check(CommonConstraints.notNull())
        vb.item(inputData.campaignPrefixName, "campaignPrefixName")
                .check(CommonConstraints.notNull())

        return vb.result
    }

    override fun execute(inputData: Param, prevState: State?): State? {
        val startRow = prevState?.lastRow ?: 0L
        val lastRow = startRow + inputData.chunkSize
        logger.info("Start from row=$startRow, to row=$lastRow (excluded)")

        val user: User = Objects.requireNonNull(userService.getUserByLogin(inputData.clientLogin))!!
        val uidAndClientId = UidAndClientId.of(user.uid, user.clientId)

        val queries = readInputTable(inputData, startRow, lastRow)
        if (queries.isEmpty()) {
            logger.info("Got empty queries, let's finish")
            return null
        }
        logger.info("queries from yt table: $queries")
        val normalizedQueries = mapList(queries, Companion::normalizeQuery)
        logger.info("normalized queries: $normalizedQueries")
        // нужно удалить невалидные ключевые фразы, т.к. операция по созданию групп не частичная:(
        val filteredQueries = filterList(normalizedQueries, Companion::isValidKeyword)

        val campaignId = addCampaign(uidAndClientId, inputData.metrikaCounters,
                inputData.meaningfulGoalIds, inputData.campaignPrefixName, startRow.toString())
        val vcardId = addVcard(campaignId, uidAndClientId)

        val adGroupIdWithQuery = addAdGroups(campaignId, uidAndClientId, filteredQueries)
        addBanners(adGroupIdWithQuery, uidAndClientId, vcardId, inputData.calloutIds, inputData.sitelinksSetId)

        if (queries.size < inputData.chunkSize) {
            logger.info("Last iteration finished")
            return null
        }
        return State(lastRow)
    }

    private fun addVcard(campaignId: Long, uidAndClientId: UidAndClientId): Long {
        val vcards = listOf(getVcard(campaignId))
        val result = vcardService.addVcardsPartial(vcards, uidAndClientId.uid, uidAndClientId.clientId)

        val successResults = getSuccessResults(result)
        logger.info("added vcardId: $successResults")
        if (successResults.isEmpty()) {
            logger.error("got vcard validationErrors: ${result.validationResult.flattenErrors()}")
        }
        checkState(successResults.size == 1, "got vcard validationErrors")
        return successResults[0]
    }

    /**
     * prefixName - нужен, чтобы имена кампаний были уникальны в рамках нескольких запусков ваншота
     * suffixName - нужен, чтобы имена кампаний были уникальны в рамках итерация одного запуска ваншота
     */
    private fun addCampaign(uidAndClientId: UidAndClientId, metrikaCounters: List<Long>,
                            meaningfulGoalIds: List<Long>, prefixName: String, suffixName: String): Long {
        val campaigns = listOf(getTextCampaign(metrikaCounters, meaningfulGoalIds, "${prefixName}_$suffixName"))
        val options = CampaignOptions.Builder()
                .withReadyToModerate(true)
                .withSkipValidateMobileApp(false)
                .withCopy(false)
                .withCopyInOneClient(false)
                .build()
        val addOperation: RestrictedCampaignsAddOperation = campaignOperationService
                .createRestrictedCampaignAddOperation(campaigns, uidAndClientId.uid,
                        UidAndClientId.of(uidAndClientId.uid, uidAndClientId.clientId), options)
        val result = addOperation.prepareAndApply()

        val successResults = getSuccessResults(result)
        logger.info("added campaignId: $successResults")
        if (successResults.isEmpty()) {
            logger.error("got campaign validationErrors: ${result.validationResult.flattenErrors()}")
        }
        checkState(successResults.size == 1, "got campaign validationErrors")
        return successResults[0]
    }

    private fun addAdGroups(campaignId: Long, uidAndClientId: UidAndClientId, queries: List<String>): Map<Long, String> {
        val adGroups = queries
                .map { getTextAdGroup(campaignId, it) }
                .toList()
        val maxPrice = Currencies.getCurrency(CurrencyCode.RUB).maxPrice
        val fixedAdGroupAutoPrices = ShowConditionFixedAutoPrices.ofGlobalFixedPrice(maxPrice)
        val showConditionAutoPriceParams = ShowConditionAutoPriceParams(fixedAdGroupAutoPrices, recentStatisticsService)
        val geoTree = clientGeoService.getClientTranslocalGeoTree(uidAndClientId.clientId)

        val addOperation = complexAdGroupAddOperationFactory
            .createTextAdGroupAddOperation(
                false, adGroups, geoTree, true,
                showConditionAutoPriceParams, uidAndClientId.uid, uidAndClientId.clientId, uidAndClientId.uid,
                DatabaseMode.ONLY_MYSQL
            )
        val result = addOperation.prepareAndApply()
        val successResults = getSuccessResults(result)
        logger.info("added adGroupIds: $successResults")

        if (result.validationResult != null && result.validationResult.hasAnyErrors()) {
            val firstErrors = result.validationResult.flattenErrors()
            logger.error("adGroups validationErrors: $firstErrors")
        }

        return StreamEx.of(adGroups)
                .map { it.adGroup.id }
                .zipWith(queries.stream())
                .filterKeys(Objects::nonNull)
                .toSortedMap()
    }

    private fun addBanners(adGroupIdWithQuery: Map<Long, String>, uidAndClientId: UidAndClientId,
                           vcardId: Long, calloutIds: List<Long>, sitelinksSetId: Long) {
        val banners = adGroupIdWithQuery
                .map { (adGroupId, query) -> getTextBanner(adGroupId, vcardId, calloutIds, sitelinksSetId, query) }
                .toList()
        val addOperation = bannersAddOperationFactory
                .createPartialAddOperation(banners, uidAndClientId.clientId, uidAndClientId.uid, false)
        val result = addOperation.prepareAndApply()
        val successResults = getSuccessResults(result)
        logger.info("added bannerIds: $successResults")

        if (result.validationResult != null && result.validationResult.hasAnyErrors()) {
            logger.error("banners validationErrors: ${result.validationResult.flattenErrors()}")

            val queriesWithValidationErrors = StreamEx.of(banners)
                    .map { it.id }
                    .zipWith(adGroupIdWithQuery.values.stream())
                    .filterKeys(Objects::isNull)
                    .values()
                    .toList()
            logger.error("got banners validationError for queries: $queriesWithValidationErrors")
        }
    }

    private fun readInputTable(inputData: Param, startRow: Long, lastRow: Long): List<String> {
        val entryType = YTableEntryTypes.yson(InputTableRow::class.java)
        val ytTable = YtTable(inputData.tablePath)

        val items = mutableListOf<String>()
        ytProvider.get(inputData.ytCluster).tables()
                .read(ytTable.ypath().withRange(startRow, lastRow), entryType,
                        Consumer { row -> items.add(row.query) })
        return items
    }

    private fun getSuccessResults(massResult: MassResult<Long>): List<Long> {
        return if (massResult.result == null) {
            emptyList()
        } else StreamEx.of(massResult.result)
                .nonNull()
                .filter { it.isSuccessful }
                .map { it.result }
                .nonNull()
                .toList()
    }

}
