package ru.yandex.direct.oneshot.oneshots.mbi

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.market.client.exception.MarketClientException
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class Param(
    val feedIds: List<Long>,
    val shouldBeEnabled: Boolean,
    val gently: Boolean
)

/**
 * Oneshot для включения/выключения фидов в MBI
 * Ограничен в 1000 фидов на запуск
 *
 * Если для включения передан Param.gently, то между отправками ждет по
 * DEFAULT_SLEEP_TIME_FOR_OFFER_IN_MS за каждый оффер фида
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov", "zakhar")
class ChangeFeedEnablingInMBIOneshot @Autowired constructor(
    private val shardHelper: ShardHelper,
    private val rbacService: RbacService,
    private val feedRepository: FeedRepository,
    private val mbiService: MbiService
) : SimpleOneshot<Param, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(ChangeFeedEnablingInMBIOneshot::class.java)
        private val relaxedWorker = RelaxedWorker()
        private const val DEFAULT_SLEEP_TIME_FOR_OFFER_IN_MS = 3
    }

    override fun validate(inputData: Param): ValidationResult<Param, Defect<*>>? {
        return ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.gently, "gently")
                .check(CommonConstraints.notNull())
            item(inputData.shouldBeEnabled, "shouldBeEnabled")
                .check(CommonConstraints.notNull())
            item(inputData.feedIds, "feeds")
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.notEmptyCollection())
                .check(CollectionConstraints.maxListSize(1000))
        }.result
    }

    override fun execute(inputData: Param?, prevState: Void?): Void? {
        logger.info("START")
        val action = if (inputData!!.shouldBeEnabled) "enable" else "disable"
        val feedIds = inputData.feedIds
        logger.info("Requested ${feedIds.size} feeds to $action")
        val feedsByShard = shardHelper.forEachShardSequential {
            return@forEachShardSequential feedRepository.getSimple(it, feedIds)
                .filterNotNull().filter { feed -> feed.marketShopId != null }
        }
        feedsByShard.forEach { (shard, feeds) ->
            if (feeds.isEmpty()) {
                logger.info("No feeds to $action on shard: $shard")
            } else {
                logger.info("${feeds.size} feeds to $action on shard: $shard")
                val feedsByClientId = feeds
                    .groupBy { feedSimple: FeedSimple -> ClientId.fromLong(feedSimple.clientId) }
                    .toMap()
                feedsByClientId.forEach { (clientId, clientFeeds) ->
                    val chiefUid = rbacService.getChiefByClientId(clientId)
                    clientFeeds.forEach {
                        if (inputData.shouldBeEnabled) {
                            enableFeed(clientId, chiefUid, it, inputData.gently)
                        } else {
                            disableFeed(clientId, chiefUid, it)
                        }
                    }
                }
            }
        }
        logger.info("FINISH")
        return null
    }

    private fun enableFeed(clientId: ClientId, chiefUid: Long, feed: FeedSimple, gently: Boolean) {
        logger.info("Enabling feed with id: ${feed.id}")
        try {
            mbiService.sendFeed(clientId, chiefUid, feed, forceEnableDirectPlacement = true)
            if (gently) {
                sleepForOffers(feed.offersCount)
            }
        } catch (e: MarketClientException) {
            logger.warn(
                "Got market client exception during enabling feed with id: ${feed.id}. " +
                    "Message: ${e.localizedMessage}"
            )
        }
    }

    private fun disableFeed(clientId: ClientId, chiefUid: Long, feed: FeedSimple) {
        logger.info("Disabling feed with id: ${feed.id}")
        try {
            relaxedWorker.runAndRelax<Exception> {
                mbiService.sendFeed(clientId, chiefUid, feed, forceEnableDirectPlacement = false)
            }
        } catch (e: MarketClientException) {
            logger.warn(
                "Got market client exception during disabling feed with id: ${feed.id}. " +
                    "Message: ${e.localizedMessage}"
            )
        }
    }

    private fun sleepForOffers(offers: Long) {
        try {
            val sleepTime = DEFAULT_SLEEP_TIME_FOR_OFFER_IN_MS * offers
            logger.info("Sleeping for $offers offers — $sleepTime milliseconds")
            Thread.sleep(sleepTime)
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
            throw RuntimeException(e)
        }
    }
}
