package ru.yandex.direct.oneshot.oneshots.mbi

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.dbschema.ppc.Tables.FEEDS
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.market.client.exception.MarketClientException
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

/**
 * Oneshot выключает в MBI фиды, которые не используются
 */
@Component
@Approvers("buhter", "kozobrodov", "zakhar")
class DisableUnusedFeedsInMBI(
    private val rbacService: RbacService,
    private val feedRepository: FeedRepository,
    private val mbiService: MbiService
) : ShardedOneshot<Void, Void> {
    private val logger = LoggerFactory.getLogger(DisableUnusedFeedsInMBI::class.java)
    private val relaxedWorker = RelaxedWorker()

    override fun validate(inputData: Void?): ValidationResult<Void, Defect<Any>> {
        return ValidationResult.success(inputData)
    }

    override fun execute(inputData: Void?, prevState: Void?, shard: Int): Void? {
        logger.info("START on shard $shard")
        val feeds = feedRepository.getSimple(
            shard, FEEDS.MARKET_SHOP_ID.isNotNull.and(
                FEEDS.USAGE_TYPE.isNull.or(FEEDS.USAGE_TYPE.eq("").or(FEEDS.LAST_USED.isNotNull))
            )
        )
        if (feeds.isEmpty()) {
            logger.info("No suitable feeds found on shard $shard")
        } else {
            logger.info("Requested ${feeds.size} feeds to disable on shard $shard")
            val feedsByClientId = feeds
                .groupBy { feedSimple: FeedSimple -> ClientId.fromLong(feedSimple.clientId) }
                .toMap()
            val chiefsByClientIds: Map<ClientId, Long> = rbacService.getChiefsByClientIds(feedsByClientId.keys)
            feedsByClientId.forEach { (clientId, clientFeeds) ->
                val uid = chiefsByClientIds[clientId]!!
                clientFeeds.forEach { feed ->
                    logger.info("Disabling feed with id: ${feed.id}")
                    try {
                        relaxedWorker.runAndRelax<Exception> {
                            mbiService.sendFeed(clientId, uid, feed)
                        }
                    } catch (e: MarketClientException) {
                        logger.warn(
                            "Got market client exception during disabling feed with id: ${feed.id}. " +
                                "Message: ${e.localizedMessage}"
                        )
                    }
                }
            }
        }
        logger.info("FINISH")
        return null
    }
}
