package ru.yandex.direct.oneshot.oneshots.mbi

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomDomainsRepository
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.utils.CollectionUtils
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import kotlin.math.min

data class EcomDomainSendingParam(
    val ecomDomains: List<String>?,
    val resendAllEcomDomains: Boolean,
    val resendWithDeleteFromMBI: Boolean
)

data class EcomDomainSendingState(
    val sent: Int,
    val offset: Int?,
    val nextEcomDomainId: Long?
)

/**
 * Oneshot на переотправку еком доменов в MBI для получения превью
 * При параметре `resendAllEcomDomains == true` - переотправляет все еком домены
 * из таблицы `ppcdict.ecom_domains`, ранее отправленные в MBI (маркетные id не null),
 * иначе только домены из списка `ecomDomains`.
 * Домены в `ecomDomains` должны быть указаны без схемы, так же, как и в `ppcdict.ecom_domains.domain`.
 *
 * Параметр `resendWithDeleteFromMBI` определяет, будут ли удалены сайты для превью из MBI, перед их повторной
 * отправкой. Если `resendWithDeleteFromMBI == false` - маркетные id будут обновлены на актуальные из таблицы в MBI,
 * иначе будут сгенерированы новые.
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers("kozobrodov", "buhter")
class ResendEcomDomainsToMBIOneshot @Autowired constructor(
    private val ecomDomainsRepository: EcomDomainsRepository,
    private val mbiService: MbiService
) : SimpleOneshot<EcomDomainSendingParam, EcomDomainSendingState> {
    companion object {
        private val logger = LoggerFactory.getLogger(ResendEcomDomainsToMBIOneshot::class.java)

        const val FETCH_LIMIT = 5_000
        private const val UPDATE_CHUNK_SIZE = 1000
    }

    override fun validate(inputData: EcomDomainSendingParam): ValidationResult<EcomDomainSendingParam, Defect<*>> {
        return ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            if (!inputData.resendAllEcomDomains) {
                item(inputData.ecomDomains, inputData::ecomDomains.name)
                    .check(CommonConstraints.notNull())
                    .check(CollectionConstraints.notEmptyCollection(), When.isValid())
            }
            val ecomDomainsIsEmpty = CollectionUtils.isEmpty(inputData.ecomDomains)
            item(inputData.resendAllEcomDomains, inputData::resendAllEcomDomains.name)
                .check(CommonConstraints.notNull())
                .check(CommonConstraints.notTrue(), When.isValidAnd(When.isFalse(ecomDomainsIsEmpty)))
        }.result
    }

    override fun execute(inputData: EcomDomainSendingParam,
                         prevState: EcomDomainSendingState?): EcomDomainSendingState? {
        logger.info("START")

        var sent = prevState?.sent ?: 0
        var offset = prevState?.offset ?: 0
        var nextEcomDomainId = prevState?.nextEcomDomainId ?: 0

        val ecomDomains = if (inputData.resendAllEcomDomains) {
            logger.info("Requested resend all ecom domains, sent to MBI")
            getSentEcomDomains(nextEcomDomainId)
        } else {
            logger.info("Requested resend ecom domains from input data")
            getSpecifiedEcomDomains(inputData.ecomDomains!!, offset)
        }

        if (ecomDomains.isEmpty() &&
            (inputData.ecomDomains == null || offset + FETCH_LIMIT >= inputData.ecomDomains.size)) {
            logger.info("Ecom domains to resend is empty")
            logger.info("Total re-sent $sent ecom domains to MBI")
            logger.info("FINISH")
            return null
        }

        logger.info("Fetched ${ecomDomains.size} ecom domains")

        if (inputData.resendWithDeleteFromMBI) {
            deleteEcomDomainsFromMbi(ecomDomains)
        }
        val appliedChanges = sendEcomDomainsToMbi(ecomDomains)
        logger.info("Re-sent ${appliedChanges.size} ecom domains")
        sent += appliedChanges.size

        updateEcomDomains(appliedChanges)

        logger.info("FINISH")
        return if (inputData.resendAllEcomDomains) {
            nextEcomDomainId = ecomDomains.last().id + 1
            EcomDomainSendingState(sent, null, nextEcomDomainId)
        } else {
            offset += FETCH_LIMIT
            EcomDomainSendingState(sent, offset, null)
        }
    }

    private fun getSpecifiedEcomDomains(ecomDomains: List<String>, offset: Int): List<EcomDomain> {
        if (offset >= ecomDomains.size) {
            return emptyList()
        }
        val ecomDomainsToProcess = ecomDomains.subList(offset, min(offset + FETCH_LIMIT, ecomDomains.size))
        if (ecomDomainsToProcess.isEmpty()) {
            return emptyList()
        }
        logger.info("Fetching ${ecomDomainsToProcess.size} ecom domains: $ecomDomainsToProcess")
        return ecomDomainsRepository.getByDomains(ecomDomainsToProcess)
    }

    private fun getSentEcomDomains(minEcomDomainId: Long): List<EcomDomain> {
        logger.info("Fetching all ecom domains sent to MBI, starting from id = $minEcomDomainId")
        return ecomDomainsRepository.getSentDomainsToMBI(minEcomDomainId, FETCH_LIMIT)
    }

    private fun deleteEcomDomainsFromMbi(ecomDomains: List<EcomDomain>) {
        try {
            logger.info("Deleting ${ecomDomains.size} site previews from MBI")
            val deleted = mbiService.deleteSitePreviews(ecomDomains)
            logger.info("Deleted $deleted site previews from MBI")
        } catch (e: Exception) {
            logger.error("Got market client exception during deleting site previews from MBI", e)
            throw RuntimeException(e)
        }
    }

    private fun sendEcomDomainsToMbi(ecomDomains: List<EcomDomain>): List<AppliedChanges<EcomDomain>> {
        val relaxedWorker = RelaxedWorker()
        return ecomDomains.map {
            try {
                val marketIds = relaxedWorker.callAndRelax<MbiService.AddSitePreviewResult, RuntimeException> {
                    mbiService.addSitePreview(it)
                }
                ecomDomainToAppliedChanges(it, marketIds)
            } catch (e: Exception) {
                logger.error("Got market client exception during sending site preview to MBI", e)
                throw RuntimeException(e)
            }
        }
    }

    private fun updateEcomDomains(appliedChanges: List<AppliedChanges<EcomDomain>>): Int {
        var updated = 0
        val chunksCount = (appliedChanges.size - 1) / UPDATE_CHUNK_SIZE + 1
        appliedChanges.chunked(UPDATE_CHUNK_SIZE)
            .forEachIndexed { index, appliedChangesChunk ->
                logger.info("Updating ${appliedChangesChunk.size} domains in DB (${index + 1} chunk of $chunksCount)")
                val chunkUpdated = ecomDomainsRepository.update(appliedChangesChunk)
                logger.info("Updated $chunkUpdated domains")

                updated += chunkUpdated
            }

        logger.info("Done. Total of $updated domains have been updated")
        return updated
    }

    private fun ecomDomainToAppliedChanges(domain: EcomDomain,
                                           marketIds: MbiService.AddSitePreviewResult): AppliedChanges<EcomDomain> {
        return ModelChanges(domain.id, EcomDomain::class.java)
            .process(marketIds.businessId, EcomDomain.MARKET_BUSINESS_ID)
            .process(marketIds.marketFeedId, EcomDomain.MARKET_FEED_ID)
            .process(marketIds.shopId, EcomDomain.MARKET_SHOP_ID)
            .applyTo(domain)
    }
}
