package ru.yandex.direct.oneshot.oneshots.mbi

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class FeaturesSendingParam(
    val feedIds: List<Long>
)

/**
 * Oneshot для ручной отправки флагов использования фидов в MBI
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov", "zakhar")
class SendFeedFeaturesToMBIOneshot @Autowired constructor(
    private val shardHelper: ShardHelper,
    private val feedRepository: FeedRepository,
    private val mbiService: MbiService
) : SimpleOneshot<FeaturesSendingParam, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(SendFeedToMBIOneshot::class.java)
    }

    override fun validate(inputData: FeaturesSendingParam): ValidationResult<FeaturesSendingParam, Defect<*>>? {
        return ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.feedIds, "feed")
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.notEmptyCollection())
        }.result
    }

    override fun execute(inputData: FeaturesSendingParam?, prevState: Void?): Void? {
        logger.info("START")
        logger.info("Requested ${inputData!!.feedIds.size} feds to send features")

        val feedIds = inputData.feedIds
        val feedsByShard = shardHelper.forEachShardSequential {
            return@forEachShardSequential feedRepository.getSimple(it, feedIds)
        }
        feedsByShard.forEach { (shard, feeds) ->
            logger.info("Start with ${feeds.size} feeds on shard: $shard")
            feeds.filterNotNull()
                .filter { feed -> feed.marketShopId?.let { it > 0 } ?: false }
                .forEach {
                    mbiService.setFeedUsageType(it.marketShopId, it.usageTypes)
                }
        }

        logger.info("FINISH")
        return null
    }
}
