package ru.yandex.direct.oneshot.oneshots.mbi

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SafeOneshot
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class FeedSendingParam(
    val feedIds: List<Long>,
    val gently: Boolean,
    val sleepTimeForOffer: Int,
    val enableDirectPlacement: Boolean?
)

data class FeedSendingState(
    val nextIndex: Int
)

/**
 * Oneshot для отправки фидов в MBI
 * Безопасный:
 * - ограничен в 1000 фидов на запуск
 * - не влияет на клиентов: нужен для прокачки фидов в MBI/DataCamp во время переезда,
 * проставляет ID клиентам недоступные
 * - может нагрузить MBI, но есть "нежная" проливка – с ней все безопасно,
 * кроме того на Директ все равно не бует доп. нагрузки
 *
 * Если передан FeedSendingParam.gently, то между отправками ждет по DEFAULT_SLEEP_TIME_FOR_OFFER_IN_MS или
 * sleepTimeForOffer за каждый оффер фида и отправляет итерациями по DEFAULT_CHUNK_SIZE
 */
@Component
@Multilaunch
@SafeOneshot
class SendFeedToMBIOneshot @Autowired constructor(
    private val shardHelper: ShardHelper,
    private val rbacService: RbacService,
    private val feedRepository: FeedRepository,
    private val mbiService: MbiService
) : SimpleOneshot<FeedSendingParam, FeedSendingState> {
    companion object {
        private val logger = LoggerFactory.getLogger(SendFeedToMBIOneshot::class.java)
        private const val DEFAULT_CHUNK_SIZE = 5
        private const val DEFAULT_SLEEP_TIME_FOR_OFFER_IN_MS = 3
    }

    override fun validate(inputData: FeedSendingParam): ValidationResult<FeedSendingParam, Defect<*>>? {
        return ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.gently, "gently")
                .check(CommonConstraints.notNull())
            item(inputData.feedIds, "feeds")
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.notEmptyCollection())
                .check(CollectionConstraints.maxListSize(1000))
        }.result
    }

    override fun execute(inputData: FeedSendingParam?, prevState: FeedSendingState?): FeedSendingState? {
        logger.info("START")
        logger.info("Requested ${inputData!!.feedIds.size} feeds to send")
        val chunkSize = if (inputData.gently) DEFAULT_CHUNK_SIZE else inputData.feedIds.size
        val feedIndexToStart = prevState?.nextIndex ?: 0
        val feedIndexToFinish = minOf(feedIndexToStart + chunkSize, inputData.feedIds.size)
        logger.info("Start index: $feedIndexToStart, batch size: $chunkSize")
        val feedIds = inputData.feedIds.subList(feedIndexToStart, feedIndexToFinish)
        val feedsByShard = shardHelper.forEachShardSequential {
            return@forEachShardSequential feedRepository.getSimple(it, feedIds)
        }
        feedsByShard.forEach { (shard, feeds) ->
            logger.info("Start with ${feeds.size} feeds on shard: $shard")
            val feedsByClientId = feeds.filterNotNull()
                .groupBy { feedSimple: FeedSimple -> ClientId.fromLong(feedSimple.clientId) }
                .toMap()
            feedsByClientId.forEach { (clientId, clientFeeds) ->
                val chiefUid = rbacService.getChiefByClientId(clientId)
                clientFeeds.forEach {
                    logger.info("Sending feed with id: ${it.id}")
                    mbiService.sendFeed(clientId, chiefUid, it, inputData.enableDirectPlacement)
                    if (inputData.gently) {
                        sleepForOffers(it.offersCount, inputData.sleepTimeForOffer)
                    }
                }
            }
        }

        logger.info("FINISH")
        return if (feedIndexToFinish == inputData.feedIds.size)
            null
        else
            FeedSendingState(feedIndexToFinish)
    }

    private fun sleepForOffers(offers: Long, sleepTimeForOffer: Int) {
        try {
            val sleepTime = offers * (
                    if (sleepTimeForOffer > 0) {
                        sleepTimeForOffer
                    } else {
                        DEFAULT_SLEEP_TIME_FOR_OFFER_IN_MS
                    })
            logger.info("Sleeping for $offers offers — $sleepTime milliseconds")
            Thread.sleep(sleepTime)
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
            throw RuntimeException(e)
        }
    }

}
