package ru.yandex.direct.oneshot.oneshots.metrika_goals_status_update

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalStatus
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsPpcDictRepository
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class MetrikaGoalsStatusUpdateState(val nextRow: Long = 0L)

data class MetrikaGoalsStatusUpdateParam(val ytCluster: YtCluster, val tablePath: String)

data class GoalIdWithStatus(val id: Long, val goalStatus: GoalStatus)

class GoalUpdateStatusTableRow : YtTableRow(listOf(ID, GOAL_STATUS)) {
    private val id: Long?
        get() = valueOf(ID)
    private val goalStatus: String?
        get() = valueOf(GOAL_STATUS)

    companion object {
        private val ID = YtField("id", Long::class.java)
        private val GOAL_STATUS = YtField("goal_status", String::class.java)

        private val logger = LoggerFactory.getLogger(MetrikaGoalsStatusUpdateOneshot::class.java)
    }

    fun convert(): GoalIdWithStatus {
        val convertedGoalStatus: GoalStatus
        try {
            convertedGoalStatus = GoalStatus.valueOf(goalStatus ?: "")
        } catch (e: IllegalStateException) {
            logger.error("Unknown goal_status=$goalStatus for goal_id=$id, skipping")
            throw e
        }
        return GoalIdWithStatus(id ?: throw IllegalStateException("ERROR: found null id"),
                convertedGoalStatus);
    }
}

@Component
@Multilaunch
@PausedStatusOnFail
@Approvers("ssdmitriev", "buhter", "pavryabov", "xy6er")
class MetrikaGoalsStatusUpdateOneshot(
        private val ytProvider: YtProvider,
        private val retargetingGoalsPpcDictRepository: RetargetingGoalsPpcDictRepository
) : SimpleOneshot<MetrikaGoalsStatusUpdateParam, MetrikaGoalsStatusUpdateState> {
    companion object {
        private const val CHUNK_SIZE = 5_000L
        private val logger: Logger = LoggerFactory.getLogger(MetrikaGoalsStatusUpdateOneshot::class.java)
    }

    override fun validate(inputData: MetrikaGoalsStatusUpdateParam): ValidationResult<MetrikaGoalsStatusUpdateParam, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(ytProvider, vb, inputData.ytCluster, inputData.tablePath)
    }

    override fun execute(inputData: MetrikaGoalsStatusUpdateParam, prevState: MetrikaGoalsStatusUpdateState?): MetrikaGoalsStatusUpdateState? {

        //если предыдущая стадия не инициализирована -- начали ваншот
        if (prevState == null) {
            logger.info("First iteration!")
            return MetrikaGoalsStatusUpdateState(0)
        }

        val startRow = prevState.nextRow
        val endRow = startRow + CHUNK_SIZE

        val goalsToUpdate = getRequestsFromYt(inputData, prevState.nextRow, endRow)

        if (goalsToUpdate.isEmpty()) {
            logger.info("Last iteration!")
            return null
        }

        logger.info("Start row $startRow, end row $endRow")

        val goalIds = goalsToUpdate.map { it.id }

        val currentGoals = retargetingGoalsPpcDictRepository.getMetrikaGoalsFromPpcDict(goalIds)
        val currentGoalById = currentGoals.map { it.id to it }.toMap()

        val appliedChanges = goalsToUpdate.asSequence()
                .filter { currentGoalById.keys.contains(it.id) }
                .map { toModelChanges(it) }
                .map { it.applyTo(currentGoalById[it.id]) }
                .toList()

        retargetingGoalsPpcDictRepository.updateMetrikaGoals(appliedChanges)
        return MetrikaGoalsStatusUpdateState(endRow)
    }

    private fun toModelChanges(goalIdWithStatus: GoalIdWithStatus): ModelChanges<Goal> {
        return ModelChanges.build(goalIdWithStatus.id, Goal::class.java, Goal.STATUS, goalIdWithStatus.goalStatus);
    }

    private fun getRequestsFromYt(inputData: MetrikaGoalsStatusUpdateParam, startRow: Long, lastRow: Long)
            : List<GoalIdWithStatus> {
        val ytTable = YtTable(inputData.tablePath)
        val ytOperator = ytProvider.getOperator(inputData.ytCluster)
        val tableRow = GoalUpdateStatusTableRow()
        val items = mutableListOf<GoalIdWithStatus>()

        ytOperator.readTableByRowRange(
                ytTable,
                { row -> items.add(row.convert()) },
                tableRow,
                startRow,
                lastRow
        )
        logger.info("Got ${items.size} items from table")
        return items
    }
}
