package ru.yandex.direct.oneshot.oneshots.migrate_ecom_campaigns

import com.fasterxml.jackson.core.type.TypeReference
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToClientIdState
import ru.yandex.direct.core.entity.feature.model.FeatureState
import ru.yandex.direct.core.entity.feature.model.FeatureState.DISABLED
import ru.yandex.direct.core.entity.feature.model.FeatureState.ENABLED
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.feature.FeatureName.ECOM_UC_NEW_BACKEND_ENABLED
import ru.yandex.direct.feature.FeatureName.MIGRATING_ECOM_CAMPAIGNS_TO_NEW_BACKEND
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow


/**
 * Ваншот для миграции ecom кампаний на единый бэкенд.
 * Включает фичу на клиенте, позволяющую UpdateAdsJob удалить все баннеры,
 * и создает заявки на обновление кампаний, которые нужно мигрировать.
 *
 * В зависимости от параметра revert (true/false) включает(выключает) фичу [UC_ECOM_NEW_BACKEND_ENABLED].
 *
 * Параметр operatorUid, должен быть равен [ru.yandex.direct.jobs.uac.UpdateAdsJobUtil.migratingOperatorUid],
 * чтобы UpdateAdsJob смогла удалить все баннеры, иначе будет работать как обычное обновление.
 */
@Component
@Multilaunch
@Approvers("xy6er", "kozobrodov")
class MigrateEcomCampaignsOneshot(
    @Autowired private val uacBannerService: UacBannerService,
    @Autowired private val featureManagingService: FeatureManagingService,
    @Autowired private val uacDbDefineService: UacDbDefineService,
    @Autowired private val uacCampaignServiceHolder: UacCampaignServiceHolder,
    @Autowired private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    @Autowired private val ytProvider: YtProvider
) : SimpleOneshot<InputData, State?> {

    override fun validate(inputData: InputData) = validateObject(inputData) {
        property(inputData::ytCluster) {
            check(notNull())
        }
        property(inputData::tablePath) {
            check(notNull())
            check(
                fromPredicate(
                    { ytProvider.getOperator(inputData.ytCluster).exists(YtTable(it)) },
                    objectNotFound()
                ), When.isValid()
            )
        }
        property(inputData::operator) {
            check(validId())
        }
    }

    override fun execute(inputData: InputData, prevState: State?): State? {
        val state = prevState ?: State(0)
        val chunkSize = inputData.chunkSize ?: MIGRATE_CHUNK_SIZE
        val startRow = state.lastRow
        val lastRow = startRow + chunkSize

        logger.info("Iteration starts with row $startRow")

        val chunk = readChunk(inputData.ytCluster, inputData.tablePath, startRow, lastRow)

        if (chunk.isEmpty()) {
            logger.info("Work is done!")
            return null
        }

        val operatorUid = inputData.operator
        val revert = inputData.revert ?: false
        val dryRun = inputData.dryRun ?: false
        val campaignIdsYdb = uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(
            chunk.flatMap { it.campaignIds }
        )
        for (dataRow in chunk) {
            if (!dryRun) {
                switchFeatureForClientId(
                    operatorUid,
                    dataRow.clientId,
                    MIGRATING_ECOM_CAMPAIGNS_TO_NEW_BACKEND,
                    ENABLED
                )
                switchFeatureForClientId(
                    operatorUid,
                    dataRow.clientId,
                    ECOM_UC_NEW_BACKEND_ENABLED,
                    if (revert) DISABLED else ENABLED
                )
            }

            for (campaignId in dataRow.campaignIds) {
                val uacCampaignId = campaignIdsYdb[campaignId] ?: campaignId.toString()
                val useGrut = uacDbDefineService.useGrut(uacCampaignId)
                logger.info("UacCampaignId $uacCampaignId in ${if (useGrut) "GrUT" else "YDB"}")
                if (!dryRun) {
                    uacCampaignServiceHolder.getUacBannerService(useGrut).updateBriefSynced(uacCampaignId, false)
                    uacBannerService.updateAdsDeferred(dataRow.clientId, operatorUid, uacCampaignId)
                }
            }
        }
        logger.info("Iteration ends with row $lastRow")
        return State(lastRow)
    }

    private fun switchFeatureForClientId(
        operatorUid: Long,
        clientId: ClientId,
        featureName: FeatureName,
        state: FeatureState
    ) {
        val textId = featureName.getName()
        val newBackendSwitchParam = FeatureTextIdToClientIdState().apply {
            this.clientId = clientId
            this.state = state
            this.textId = textId
        }

        logger.info("Switch ${if (state == ENABLED) "on" else "off"} $textId for client $clientId")
        featureManagingService.switchFeaturesStateForClientIds(operatorUid, listOf(newBackendSwitchParam))
    }

    fun readChunk(ytCluster: YtCluster, tablePath: String, startRow: Long, lastRow: Long): List<DataRow> {
        val chunk = mutableListOf<DataRow>()
        ytProvider.getOperator(ytCluster).readTableByRowRange(
            YtTable(tablePath), {
                chunk += DataRow(ClientId.fromLong(it.clientId), it.campaignIds)
            },
            InputTableRow(), startRow, lastRow
        )
        return chunk
    }

    companion object {
        private val logger = LoggerFactory.getLogger(MigrateEcomCampaignsOneshot::class.java)
    }
}

data class InputData(
    val ytCluster: YtCluster,
    val tablePath: String,
    val operator: Long,
    val dryRun: Boolean?,
    val revert: Boolean?,
    val chunkSize: Int?
)

private const val MIGRATE_CHUNK_SIZE = 1

data class State(
    var lastRow: Long,
)

data class DataRow(
    val clientId: ClientId,
    val campaignIds: List<Long>
)

class InputTableRow : YtTableRow(listOf(CLIENT_ID, CAMPAIGN_IDS)) {
    companion object {
        private val CLIENT_ID = YtField("client_id", Long::class.java)
        private val CAMPAIGN_IDS = YtField("campaign_ids", String::class.java)
    }

    val campaignIds: List<Long>
        get() = JsonUtils.fromJson(valueOf(CAMPAIGN_IDS), object : TypeReference<List<Long>>() {})

    val clientId: Long
        get() = valueOf(CLIENT_ID)
}
