package ru.yandex.direct.oneshot.oneshots.mobile_app_tracker

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.MobileAppTrackersTrackingSystem
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.oneshots.mobilecontent.Param
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SafeOneshot
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeField
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeObject
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes
import java.util.function.Consumer

data class Param(
    val ytCluster: YtCluster,
    val tablePath: String
)

@YTreeObject
data class InputTableRow(
    @YTreeField(key = "mobile_app_tracker_id") val mobileAppTrackerId: Long?,
    @YTreeField(key = "client_id") val clientId: Long,
    @YTreeField(key = "tracking_system") val trackingSystem: String,
)

/**
 * Ваншот для обновления трекерной системы в мобильных трекерах.
 */
@Component
@Multilaunch
@SafeOneshot
@PausedStatusOnFail
@Retries(2)
class UpdateTrackingSystemForMobileApps @Autowired constructor(
    private val ytProvider: YtProvider,
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper,
) : SimpleOneshot<Param, Void> {
    override fun validate(inputData: Param): ValidationResult<Param, Defect<*>>? {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)

        vb.item(inputData.ytCluster, "ytCluster")
            .check(CommonConstraints.notNull())

        if (vb.result.hasAnyErrors()) return vb.result

        vb.item(inputData.tablePath, "tablePath")
            .check(CommonConstraints.notNull())
            .check(
                Constraint.fromPredicate(
                    { tableName -> ytProvider.getOperator(inputData.ytCluster).exists(YtTable(tableName)) },
                    CommonDefects.objectNotFound()
                )
            )
        return vb.result
    }

    override fun execute(inputData: Param, prevState: Void?): Void? {
        val updateTrackingSystemItems = readInputTable(inputData)

        for (updateTrackingSystemItem in updateTrackingSystemItems) {
            updateMobileAppTrackersTrackingSystem(
                updateTrackingSystemItem
            )
        }

        return null
    }

    private fun updateMobileAppTrackersTrackingSystem(
        updateTrackingSystemItem: UpdateTrackingSystemParam
    ) {
        val shard = shardHelper.getShardByClientId(ClientId.fromLong(updateTrackingSystemItem.clientId))
        dslContextProvider.ppc(shard)
            .update(Tables.MOBILE_APP_TRACKERS)
            .set(
                Tables.MOBILE_APP_TRACKERS.TRACKING_SYSTEM,
                MobileAppTrackersTrackingSystem.valueOf(updateTrackingSystemItem.trackingSystem)
            )
            .where(Tables.MOBILE_APP_TRACKERS.MOBILE_APP_TRACKER_ID.eq(updateTrackingSystemItem.mobileAppTrackerId))
            .execute()
    }

    private fun readInputTable(inputData: Param): MutableList<UpdateTrackingSystemParam> {
        val entryType = YTableEntryTypes.yson(InputTableRow::class.java)
        val ytTable = YtTable(inputData.tablePath)

        val items = mutableListOf<UpdateTrackingSystemParam>()
        ytProvider.get(inputData.ytCluster).tables()
            .read(ytTable.ypath(), entryType,
                Consumer { row ->
                    items.add(UpdateTrackingSystemParam(row.mobileAppTrackerId!!, row.trackingSystem, row.clientId))
                })
        return items
    }

    inner class UpdateTrackingSystemParam(
        val mobileAppTrackerId: Long,
        val trackingSystem: String,
        val clientId: Long
    )
}
