package ru.yandex.direct.oneshot.oneshots.mobilecontent

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeField
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeObject
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes
import java.util.function.Consumer

data class Param(
    val ytCluster: YtCluster,
    val tablePath: String
)

data class State(
    val lastRow: Long = 0L
)

@YTreeObject
data class InputTableRow(
    @YTreeField(key = "mobile_content_id") val mobileContentId: Long?
)

/**
 * Ваншот для сброса статуса синхронизации с БК (statusBsSynced) мобильных приложений (mobile_content).
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers("pavryabov", "xy6er", "zakhar")
class MobileContentResetStatusBsSyncedOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val mobileContentRepository: MobileContentRepository
) : ShardedOneshot<Param, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(MobileContentResetStatusBsSyncedOneshot::class.java)
        private const val CHUNK_SIZE = 5000L
    }

    override fun validate(inputData: Param): ValidationResult<Param, Defect<*>>? {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)

        vb.item(inputData.ytCluster, "ytCluster")
            .check(notNull())

        if (vb.result.hasAnyErrors()) return vb.result

        vb.item(inputData.tablePath, "tablePath")
            .check(notNull())
            .check(Constraint.fromPredicate(
                { tableName -> ytProvider.getOperator(inputData.ytCluster).exists(YtTable(tableName)) },
                objectNotFound()))
        return vb.result
    }

    override fun execute(inputData: Param, prevState: State?, shard: Int): State? {
        val startRow = prevState?.lastRow ?: 0L
        val lastRow = startRow + CHUNK_SIZE
        logger.info("Shard=$shard: Start from row=$startRow, to row=$lastRow (excluded)")

        val mobileContentIds = readInputTable(inputData, startRow, lastRow)

        val changed = mobileContentRepository.changeStatusBsSynced(
            shard, StatusBsSynced.YES, StatusBsSynced.NO, mobileContentIds)
        logger.info("Shard=$shard: statusBsSynced of $changed records were changed during iteration")

        if (mobileContentIds.size < CHUNK_SIZE) {
            logger.info("Shard=$shard: Last iteration finished")
            return null
        }
        return State(lastRow)
    }

    private fun readInputTable(inputData: Param, startRow: Long, lastRow: Long): List<Long> {
        val entryType = YTableEntryTypes.yson(InputTableRow::class.java)
        val ytTable = YtTable(inputData.tablePath)

        val items = mutableListOf<Long>()
        ytProvider.get(inputData.ytCluster).tables()
            .read(ytTable.ypath().withRange(startRow, lastRow), entryType,
                Consumer { row -> items.add(row.mobileContentId!!) })
        return items
    }
}
