package ru.yandex.direct.oneshot.oneshots.mrcfrontpage;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;

/**
 * Эксперимент по включению стандарта определения видимости MRC
 * для кампании с типом Медийная на Главной.
 * https://st.yandex-team.ru/DIRECT-118337
 * <p>
 * Ваншот ожидает входной параметр - строку с campaignId, которой нужно выставить параметр.
 * Кампания должна иметь тип cpm_yndx_frontpage
 */
@ParametersAreNonnullByDefault
@Component
@Multilaunch
@Approvers({"ivatkoegor", "hrustyashko"})
public class MRCForYndxFrontpageExperimentDirect118337 implements SimpleOneshot<String, Void> {
    private static final Logger logger = LoggerFactory.getLogger(MRCForYndxFrontpageExperimentDirect118337.class);

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    public MRCForYndxFrontpageExperimentDirect118337(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    @Nullable
    @Override
    public Void execute(@Nullable String inputData, @Nullable Void prevState) {
        checkNotNull(inputData);
        Long campaignId = Long.parseLong(inputData);
        int shard = shardHelper.getShardByCampaignId(campaignId);

        Long oldValue = dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.IMPRESSION_STANDARD_TIME)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.eq(campaignId))
                .fetchOne(CAMP_OPTIONS.IMPRESSION_STANDARD_TIME);
        logger.info("Old impression_standard_time for campaign {} is {}", campaignId, oldValue);

        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.IMPRESSION_STANDARD_TIME, 1000L)
                .where(CAMP_OPTIONS.CID.eq(campaignId))
                .execute();
        return null;
    }

    @Override
    public ValidationResult<String, Defect> validate(String inputData) {
        Long campaignId = Long.parseLong(inputData);
        int shard = shardHelper.getShardByCampaignId(campaignId);
        if (shard == NO_SHARD) {
            logger.error("Campaign {} not found", campaignId);
            return ValidationResult.failed(inputData, CommonDefects.invalidValue());
        }
        CampaignsType campType = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.TYPE).from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.eq(campaignId)).fetchOne(CAMPAIGNS.TYPE);
        if (campType != CampaignsType.cpm_yndx_frontpage) {
            logger.error("Campaign {} has wrong type {}", campaignId, campType);
            return ValidationResult.failed(inputData, CommonDefects.invalidValue());
        }

        return ValidationResult.success(inputData);
    }
}
