package ru.yandex.direct.oneshot.oneshots.mysql2grut

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository
import ru.yandex.direct.core.mysql2grut.repository.MinusPhraseRepository
import ru.yandex.direct.dbschema.ppc.Tables.BIDS
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_BASE
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING
import ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS
import ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES
import ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.ess.config.mysql2grut.Mysql2GrutReplicationConfig
import ru.yandex.direct.ess.logicobjects.mysql2grut.BidModifierTableType
import ru.yandex.direct.ess.logicobjects.mysql2grut.BiddableShowConditionChangeType
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.SupportedCampaignsService
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class State(
    val lastRow: Long
)

data class ObjectIds(
    val clientId: Long? = null,
    val campaignId: Long? = null,
    val adGroupId: Long? = null,
    val bannerId: Long? = null,
    val minusPhraseId: Long? = null,
    val mobileContentId: Long? = null,
    val vcardId: Long? = null,
    val bidModifierId: Long? = null,
    val creative_id: Long? = null,
    val retargetingConditionId: Long? = null,
    val keywordId: Long? = null,
    val offerRetargetingId: Long? = null,
    val relevanceMatchId: Long? = null,
    val dynamicBidId: Long? = null,
    val performanceBidId: Long? = null,
    val retargetingBidId: Long? = null,
    val strategyId: Long? = null,
    val predefinedShard: Int? = null,
)

class InputTableRow : YtTableRow(ALL_COLUMNS) {
    companion object {
        val CLIENT_ID = YtField("client_id", Long::class.java)
        val CAMPAIGN_ID = YtField("campaign_id", Long::class.java)
        val ADGROUP_ID = YtField("adgroup_id", Long::class.java)
        val BANNER_ID = YtField("banner_id", Long::class.java)
        val MINUS_PHRASE_ID = YtField("minus_phrase_id", Long::class.java)
        val MOBILE_CONTENT_ID = YtField("mobile_content_id", Long::class.java)
        val VCARD_ID = YtField("vcard_id", Long::class.java)
        val BID_MODIFIER_ID = YtField("bid_modifier_id", Long::class.java)
        val CREATIVE_ID = YtField("creative_id", Long::class.java)
        val RETARGETING_CONDITION_ID = YtField("retargeting_condition_id", Long::class.java)
        val KEYWORD_ID = YtField("keyword_id", Long::class.java)
        val OFFER_RETARGETING_ID = YtField("offer_retargeting_id", Long::class.java)
        val RELEVANCE_MATCH_ID = YtField("relevance_match_id", Long::class.java)
        val DYNAMIC_BID_ID = YtField("dynamic_bid_id", Long::class.java)
        val PERFORMANCE_BID_ID = YtField("performance_bid_id", Long::class.java)
        val RETARGETING_BID_ID = YtField("retargeting_bid_id", Long::class.java)
        val STRATEGY_ID = YtField("strategy_id", Long::class.java)

        val ALL_COLUMNS = listOf(
            CLIENT_ID,
            CAMPAIGN_ID,
            ADGROUP_ID,
            BANNER_ID,
            MINUS_PHRASE_ID,
            MOBILE_CONTENT_ID,
            VCARD_ID,
            BID_MODIFIER_ID,
            CREATIVE_ID,
            RETARGETING_CONDITION_ID,
            KEYWORD_ID,
            OFFER_RETARGETING_ID,
            RELEVANCE_MATCH_ID,
            DYNAMIC_BID_ID,
            PERFORMANCE_BID_ID,
            RETARGETING_BID_ID,
            STRATEGY_ID,
        )
    }

    fun shard(columnName: String?): Int? {
        if (columnName == null) {
            return null
        }

        val column = data[columnName]
        if (column.isEmpty) {
            return null
        }

        return column.get().intValue()
    }

    val clientId: Long?
        get() = valueOf(CLIENT_ID)

    val campaignId: Long?
        get() = valueOf(CAMPAIGN_ID)

    val adGroupId: Long?
        get() = valueOf(ADGROUP_ID)

    val bannerId: Long?
        get() = valueOf(BANNER_ID)

    val minusPhraseId: Long?
        get() = valueOf(MINUS_PHRASE_ID)

    val mobileContentId: Long?
        get() = valueOf(MOBILE_CONTENT_ID)

    val vcardId: Long?
        get() = valueOf(VCARD_ID)

    val bidModifierId: Long?
        get() = valueOf(BID_MODIFIER_ID)

    val creativeId: Long?
        get() = valueOf(CREATIVE_ID)

    val retargetingConditionId: Long?
        get() = valueOf(RETARGETING_CONDITION_ID)

    val keywordId: Long?
        get() = valueOf(KEYWORD_ID)

    val offerRetargetingId: Long?
        get() = valueOf(OFFER_RETARGETING_ID)

    val relevanceMatchId: Long?
        get() = valueOf(RELEVANCE_MATCH_ID)

    val dynamicBidId: Long?
        get() = valueOf(DYNAMIC_BID_ID)

    val performanceBidId: Long?
        get() = valueOf(PERFORMANCE_BID_ID)

    val retargetingBidId: Long?
        get() = valueOf(RETARGETING_BID_ID)

    val strategyId: Long?
        get() = valueOf(STRATEGY_ID)
}

/**
 * Ваншот для переотправки данных в репликацию mysql2grut
 * Поддержано несколько типов объектов, смотри InputTableRow.

 * Для запуска ваншота нужно подготовить таблицу на yt с колонкой client_id (или другой, соответствующей объекту)
 * Для оптимальной работы следует отсортировать по шарду
 * Передать таблицу в параметрах ваншота `{"tablePath": "//home/direct/test/mspirit/clients_for_resync", "ytCluster": "HAHN"}`
 *
 * Для ускорения работы ваншота в исходной таблице можно сделать какую-нибудь числовую колонку с номером шарда,
 * и добавить имя этой колонки в качестве параметра ваншота shardColumnName. При указании этого параметра валидация
 * проверит, что в первой строке таблицы есть указанная колонка и что её тип — числовой
 *
 * Кроме этого важно понимать, что работа ваншота влияет на скорость самой репликации, так как ваншот только кладет нужные объекты в очередь ess
 * Поэтому при запуске ваншота важно следить за скоростью самой реплиуации, например, по графикам https://nda.ya.ru/t/GE6X8K4g4hLVLE
 * Если скрорость ваншота слишком большая или маленькая, его можно регулировать пропертями
 * - mysql2grut_oneshot_chunk_size - количество прочитанных строк за итерацию
 * - mysql2grut_oneshot_relax_time_sec - время отдыха между итерациями, если за время ее работы какие то объекты были отправлены
 */
@Component
@Approvers("mspirit", "ppalex", "elwood")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class Mysql2GrutReplicationOneshot(
    private val ytProvider: YtProvider,
    private val essClient: EssClient,
    private val shardHelper: ShardHelper,
    private val supportedCampaignsService: SupportedCampaignsService,
    private val adGroupRepository: AdGroupRepository,
    private val bannerRepository: BannerTypedRepository,
    private val minusPhraseRepository: MinusPhraseRepository,
    private val vcardRepository: VcardRepository,
    private val mobileContentRepository: MobileContentRepository,
    private val dslContextProvider: DslContextProvider,
    private val strategyTypedRepository: StrategyTypedRepository,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : ShardedOneshot<ShardedParam, State?> {

    private val logicProcessorName = Mysql2GrutReplicationConfig().logicProcessName

    private val chunkSizeProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.MYSQL2GRUT_ONESHOT_CHUNK_SIZE)

    private val relaxTimeProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.MYSQL2GRUT_ONESHOT_RELAX_TIME_SEC)

    private val ytUtils = Mysql2GrutYtUtils(ytProvider)

    companion object {
        private val logger = LoggerFactory.getLogger(Mysql2GrutReplicationOneshot::class.java)
    }

    override fun validate(inputData: ShardedParam) = ytUtils.validate(inputData)

    override fun execute(inputData: ShardedParam, prevState: State?, shard: Int): State? {
        val chunkSize = chunkSizeProperty.getOrDefault(100)
        val relaxTime = relaxTimeProperty.getOrDefault(60)
        logger.info("Start from state=$prevState, shard=$shard with chunkSize=$chunkSize, relaxTime = $relaxTime")

        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val objectIds = getIdsToSend(inputData, startRow, lastRow)
        val handledClients = sendClients(shard, objectIds)
        val handledCampaigns = sendCampaigns(shard, objectIds)
        val handledAdGroups = sendAdGroups(shard, objectIds)
        val handledBanners = sendBanners(shard, objectIds)
        val handledMinusPhrases = sendMinusPhrases(shard, objectIds)
        val handledMobileContents = sendMobileContents(shard, objectIds)
        val handledVcards = sendVcards(shard, objectIds)
        val handledBidModifiers = sendBidModifiers(shard, objectIds)
        val handledCreatives = sendCreatives(shard, objectIds)
        val handledRetargetingConditions = sendRetargetingConditions(shard, objectIds)
        val handledKeywords = sendKeywords(shard, objectIds)
        val handledOfferRetargeting = sendOfferRetargeting(shard, objectIds)
        val handledRelevanceMatch = sendRelevanceMatch(shard, objectIds)
        val handledDynamicBids = sendDynamicBids(shard, objectIds)
        val handledPerformanceBids = sendPerformanceBids(shard, objectIds)
        val handledRetargetingBids = sendRetargetingBids(shard, objectIds)
        val handledStrategies = sendStrategies(shard, objectIds)

        val handledObjects = (handledClients + handledCampaigns + handledAdGroups + handledBanners +
            handledMinusPhrases + handledMobileContents + handledMinusPhrases + handledVcards + handledBidModifiers +
            handledCreatives + handledRetargetingConditions + handledKeywords + handledOfferRetargeting +
            handledRelevanceMatch + handledDynamicBids + handledPerformanceBids + handledRetargetingBids + handledStrategies)

        if (handledObjects != 0) {
            logger.info("Shard=$shard: Iteration finished, sleep for $relaxTime seconds")
            Thread.sleep(relaxTime * 1000)
        } else {
            logger.info("No handled objects on iteration")
        }
        return if (objectIds.size < chunkSize) null else State(lastRow)
    }

    private fun predefinedShardFilter(obj: ObjectIds, shard: Int) =
        obj.predefinedShard == null || obj.predefinedShard == shard

    private fun sendClients(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val clientsIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.clientId }
            .distinct()
        if (clientsIds.isEmpty()) {
            return 0
        }
        val clientsObjects = shardHelper.getShardsByClientIds(clientsIds)
            .filter { it.value == shard }
            .map { Mysql2GrutReplicationObject(clientId = it.key) }
        logger.info("Going to send ${clientsObjects.size} clients")
        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, clientsObjects)
        return clientsObjects.size
    }

    private fun sendCampaigns(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val campaignsIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.campaignId }
            .distinct()
        if (campaignsIds.isEmpty()) {
            return 0
        }
        val campaignObjects = supportedCampaignsService.getSafely(shard, campaignsIds, CommonCampaign::class.java)
            .map { Mysql2GrutReplicationObject(campaignId = it.id) }
        logger.info("Going to send ${campaignObjects.size} campaigns")
        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, campaignObjects)
        return campaignObjects.size
    }

    private fun sendAdGroups(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val adGroupIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.adGroupId }
            .toSet()
        if (adGroupIds.isEmpty()) {
            return 0
        }
        val selectionCriteria = AdGroupsSelectionCriteria().withAdGroupIds(adGroupIds)
        val adGroupObjects =
            adGroupRepository.getAdGroupIdsBySelectionCriteria(shard, selectionCriteria, LimitOffset.maxLimited())
                .map { Mysql2GrutReplicationObject(adGroupId = it) }
        logger.info("Going to send ${adGroupObjects.size} adGroups")
        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, adGroupObjects)
        return adGroupObjects.size
    }

    private fun sendBanners(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val bannerIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.bannerId }
            .toSet()
        if (bannerIds.isEmpty()) {
            return 0
        }
        val bannerObjects = bannerRepository.getSafely(shard, bannerIds, BannerWithSystemFields::class.java)
            .map { Mysql2GrutReplicationObject(bannerId = it.id) }

        logger.info("Going to send ${bannerObjects.size} banners")
        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, bannerObjects)
        return bannerObjects.size
    }

    private fun sendMinusPhrases(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val minusPhrasesIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.minusPhraseId }
            .toSet()
        if (minusPhrasesIds.isEmpty()) {
            return 0
        }
        val minusPhrasesObjects = minusPhraseRepository.getMinusPhrases(shard, minusPhrasesIds)
            .map { Mysql2GrutReplicationObject(minusPhraseId = it.id) }

        logger.info("Going to send ${minusPhrasesObjects.size} minus phrases")
        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, minusPhrasesObjects)
        return minusPhrasesObjects.size
    }

    private fun sendMobileContents(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val mobileContentsIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.mobileContentId }
            .toSet()
        if (mobileContentsIds.isEmpty()) {
            return 0
        }
        val mobileContentObjects = mobileContentRepository.getMobileContent(shard, mobileContentsIds)
            .map { Mysql2GrutReplicationObject(mobileContentId = it.id) }

        logger.info("Going to send ${mobileContentObjects.size} mobile contents")
        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, mobileContentObjects)
        return mobileContentObjects.size
    }

    private fun sendVcards(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val vcardIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.vcardId }
            .toSet()
        if (vcardIds.isEmpty()) {
            return 0
        }
        val vcardsObjects =
            vcardRepository.selectVcardsWithoutPoints(shard, null, vcardIds, null, LimitOffset.maxLimited())
                .map { Mysql2GrutReplicationObject(vcardId = it.id) }

        logger.info("Going to send ${vcardsObjects.size} vcards")
        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, vcardsObjects)
        return vcardsObjects.size
    }

    private fun sendBidModifiers(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val bidModifiersIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.bidModifierId }
            .toSet()
        if (bidModifiersIds.isEmpty()) {
            return 0
        }

        val bidModifiersObjects = dslContextProvider.ppc(shard)
            .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
            .from(HIERARCHICAL_MULTIPLIERS)
            .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.`in`(bidModifiersIds))
            .fetch(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
            // всегда тип указываем PARENT, по нему в процессоре подтянутся все нужные поля
            .map { Mysql2GrutReplicationObject(bidModifierId = it, bidModifierTableType = BidModifierTableType.PARENT) }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, bidModifiersObjects)
        return bidModifiersObjects.size
    }

    private fun sendCreatives(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val creativeIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.creative_id }
            .toSet()
        if (creativeIds.isEmpty()) {
            return 0
        }

        val creativeObjects = dslContextProvider.ppc(shard)
            .select(PERF_CREATIVES.CREATIVE_ID)
            .from(PERF_CREATIVES)
            .where(PERF_CREATIVES.CREATIVE_ID.`in`(creativeIds))
            .fetch(PERF_CREATIVES.CREATIVE_ID)
            .map { Mysql2GrutReplicationObject(creativeId = it) }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, creativeObjects)
        return creativeObjects.size
    }

    private fun sendRetargetingConditions(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val retCondIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.retargetingConditionId }
            .toSet()
        if (retCondIds.isEmpty()) {
            return 0
        }

        val retCondObjects = dslContextProvider.ppc(shard)
            .select(RETARGETING_CONDITIONS.RET_COND_ID)
            .from(RETARGETING_CONDITIONS)
            .where(RETARGETING_CONDITIONS.RET_COND_ID.`in`(retCondIds))
            .fetch(RETARGETING_CONDITIONS.RET_COND_ID)
            .map { Mysql2GrutReplicationObject(retargetingConditionId = it) }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, retCondObjects)
        return retCondObjects.size
    }

    private fun extractIds(shard: Int, objectIdsList: List<ObjectIds>, extract: (ObjectIds) -> Long?): Set<Long> {
        return objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { extract(it) }
            .toSet()
    }

    private fun sendKeywords(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val keywordIds = extractIds(shard, objectIdsList) { obj -> obj.keywordId }
        if (keywordIds.isEmpty()) return 0

        val replicationObjects = dslContextProvider.ppc(shard)
            .select(BIDS.ID)
            .from(BIDS)
            .where(BIDS.ID.`in`(keywordIds))
            .fetch(BIDS.ID)
            .map {
                Mysql2GrutReplicationObject(
                    biddableShowConditionId = it,
                    biddableShowConditionType = BiddableShowConditionChangeType.KEYWORD,
                )
            }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, replicationObjects)
        return replicationObjects.size
    }

    private fun sendOfferRetargeting(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val offerRetargetingIds = extractIds(shard, objectIdsList) { obj -> obj.offerRetargetingId }
        if (offerRetargetingIds.isEmpty()) return 0

        val replicationObjects = dslContextProvider.ppc(shard)
            .select(BIDS_BASE.BID_ID)
            .from(BIDS_BASE)
            .where(BIDS_BASE.BID_ID.`in`(offerRetargetingIds))
            .and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
            .fetch(BIDS_BASE.BID_ID)
            .map {
                Mysql2GrutReplicationObject(
                    biddableShowConditionId = it,
                    biddableShowConditionType = BiddableShowConditionChangeType.OFFER_RETARGETING,
                )
            }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, replicationObjects)
        return replicationObjects.size
    }

    private fun sendRelevanceMatch(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val relevanceMatchIds = extractIds(shard, objectIdsList) { obj -> obj.relevanceMatchId }
        if(relevanceMatchIds.isEmpty()) return 0

        val replicationObjects = dslContextProvider.ppc(shard)
            .select(BIDS_BASE.BID_ID)
            .from(BIDS_BASE)
            .where(BIDS_BASE.BID_ID.`in`(relevanceMatchIds))
            .and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.relevance_match))
            .fetch(BIDS_BASE.BID_ID)
            .map {
                Mysql2GrutReplicationObject(
                    biddableShowConditionId = it,
                    biddableShowConditionType = BiddableShowConditionChangeType.RELEVANCE_MATCH,
                )
            }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, replicationObjects)
        return replicationObjects.size
    }

    private fun sendDynamicBids(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val dynamicBidsIds = extractIds(shard, objectIdsList) { obj -> obj.dynamicBidId }
        if(dynamicBidsIds.isEmpty()) return 0

        val replicationObjects = dslContextProvider.ppc(shard)
            .select(BIDS_DYNAMIC.DYN_ID)
            .from(BIDS_DYNAMIC)
            .where(BIDS_DYNAMIC.DYN_ID.`in`(dynamicBidsIds))
            .fetch(BIDS_DYNAMIC.DYN_ID)
            .map {
                Mysql2GrutReplicationObject(
                    biddableShowConditionId = it,
                    biddableShowConditionType = BiddableShowConditionChangeType.DYNAMIC,
                )
            }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, replicationObjects)
        return replicationObjects.size
    }

    private fun sendPerformanceBids(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val performanceBidsIds = extractIds(shard, objectIdsList) { obj -> obj.performanceBidId }
        if(performanceBidsIds.isEmpty()) return 0

        val replicationObjects = dslContextProvider.ppc(shard)
            .select(BIDS_PERFORMANCE.PERF_FILTER_ID)
            .from(BIDS_PERFORMANCE)
            .where(BIDS_PERFORMANCE.PERF_FILTER_ID.`in`(performanceBidsIds))
            .fetch(BIDS_PERFORMANCE.PERF_FILTER_ID)
            .map {
                Mysql2GrutReplicationObject(
                    biddableShowConditionId = it,
                    biddableShowConditionType = BiddableShowConditionChangeType.PERFORMANCE,
                )
            }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, replicationObjects)
        return replicationObjects.size
    }

    private fun sendRetargetingBids(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val retargetingBidsIds = extractIds(shard, objectIdsList) { obj -> obj.retargetingBidId }
        if(retargetingBidsIds.isEmpty()) return 0

        val replicationObjects = dslContextProvider.ppc(shard)
            .select(BIDS_RETARGETING.RET_ID)
            .from(BIDS_RETARGETING)
            .where(BIDS_RETARGETING.RET_ID.`in`(retargetingBidsIds))
            .fetch(BIDS_RETARGETING.RET_ID)
            .map {
                Mysql2GrutReplicationObject(
                    biddableShowConditionId = it,
                    biddableShowConditionType = BiddableShowConditionChangeType.RETARGETING,
                )
            }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, replicationObjects)
        return replicationObjects.size
    }

    private fun sendStrategies(shard: Int, objectIdsList: List<ObjectIds>): Int {
        val strategiesIds = objectIdsList
            .filter { predefinedShardFilter(it, shard) }
            .mapNotNull { it.strategyId }
            .toSet()
        if (strategiesIds.isEmpty()) {
            return 0
        }

        val strategiesObjects = strategyTypedRepository.getSafely(shard, strategiesIds, CommonStrategy::class.java)
            .map { Mysql2GrutReplicationObject(strategyId = it.id) }

        essClient.addLogicObjectsForProcessor(shard, logicProcessorName, strategiesObjects)
        return strategiesObjects.size
    }

    private fun getIdsToSend(inputData: ShardedParam, startRow: Long, lastRow: Long): List<ObjectIds> {
        val objectIds = mutableListOf<ObjectIds>()
        ytProvider.getOperator(inputData.ytCluster)
            .readTableByRowRange(YtTable(inputData.tablePath), {
                objectIds.add(
                    ObjectIds(
                        clientId = it.clientId,
                        campaignId = it.campaignId,
                        adGroupId = it.adGroupId,
                        bannerId = it.bannerId,
                        minusPhraseId = it.minusPhraseId,
                        mobileContentId = it.mobileContentId,
                        vcardId = it.vcardId,
                        bidModifierId = it.bidModifierId,
                        creative_id = it.creativeId,
                        retargetingConditionId = it.retargetingConditionId,
                        keywordId = it.keywordId,
                        offerRetargetingId = it.offerRetargetingId,
                        relevanceMatchId = it.relevanceMatchId,
                        dynamicBidId = it.dynamicBidId,
                        performanceBidId = it.performanceBidId,
                        retargetingBidId = it.retargetingBidId,
                        strategyId = it.strategyId,
                        predefinedShard = it.shard(inputData.shardColumnName),
                    )
                )
            }, InputTableRow(), startRow, lastRow)
        return objectIds
    }
}
