package ru.yandex.direct.oneshot.oneshots.mysql2grut.delete_objects

import org.jooq.Record
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING
import ru.yandex.direct.dbschema.ppc.tables.BidsRetargeting
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.ess.config.mysql2grut.Mysql2GrutReplicationConfig
import ru.yandex.direct.ess.logicobjects.mysql2grut.BiddableShowConditionChangeType
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.oneshot.oneshots.mysql2grut.Mysql2GrutYtUtils
import ru.yandex.direct.oneshot.oneshots.mysql2grut.Param
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

/**
 * Ваншот удаление дублей: дубль - несколько записей с одинаковым идентификатором в нескольких шардах.
 * На вход передаем идентификаторы записей, ваншот проверяет, что это действительно дубль и удаляет ту запись,
 * которая не отправляется транспортом в Цезарь.
 * Дополнительно объект отправляется на удаление из грута через репликацию.
 */
@Component
@Approvers("mspirit", "ppalex", "elwood")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class DeleteDublicatesFromMysqlOneshot(
    private val ytProvider: YtProvider,
    private val shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
    private val essClient: EssClient,
) : SimpleOneshot<Param, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(DeleteDublicatesFromMysqlOneshot::class.java)
        const val chunkSize = 100
        const val relaxTime = 10L
    }

    private val ytUtils = Mysql2GrutYtUtils(ytProvider)
    private val logicProcessorName = Mysql2GrutReplicationConfig().logicProcessName

    override fun validate(inputData: Param) = ytUtils.validate(inputData)

    override fun execute(inputData: Param, prevState: State?): State? {

        logger.info("Start from state=$prevState, with chunkSize=$chunkSize, relaxTime = $relaxTime")
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val objectIds = readIdsToDedup(inputData.ytCluster, inputData.tablePath, startRow, lastRow)

        val dedupedObjects = dedupBidRetargetings(objectIds)
        if (dedupedObjects != 0) {
            logger.info("Iteration finished, sleep for $relaxTime seconds")
            Thread.sleep(relaxTime * 1000)
        } else {
            logger.info("No handled objects on iteration")
        }
        return if (objectIds.size < chunkSize) null else State(lastRow)
    }

    private fun dedupBidRetargetings(objectIdsList: List<DedupObjectIds>): Int {
        val retIds = objectIdsList.mapNotNull { it.retargetingBidId }.toSet()
        val shardsToDeleteFrom = objectIdsList.associate { it.retargetingBidId to it.shard }

        val duplicates = readRetargetingBidsDuplicates(retIds)

        var deleted: Int = 0
        if (retIds.size != duplicates.size) {
            logger.warn(
                "${retIds.size} retargeting_bid ids with duplicates passed, but ${duplicates.size} objects found, " +
                    "missing ids: ${retIds.minus(duplicates.keys)}"
            )
        }
        for ((retId, retObjects) in duplicates) {
            when (retObjects.size) {
                // unexpected: no duplicates
                1 -> {
                    logger.warn("bidRetargeting $retId has only one record in shard ${retObjects.first().shard}")
                }
                // when two duplicates found we delete row with "Sending" status.
                // do additional check duplicate.shard == passed shard
                2 -> {
                    val notSyncedObject = retObjects.filter { it.statusBsSynced == StatusBsSynced.SENDING }
                    if (notSyncedObject.size == 1 && notSyncedObject.first().shard == shardsToDeleteFrom[retId]) {
                        deleteBidRetargetingDuplicate(notSyncedObject.first())
                        deleted++
                    } else {
                        logger.warn("SKIPPING bidRetargeting $retId: ${notSyncedObject.size} records with SENDING status")
                    }
                }
                // unexpected: more than two duplicates
                else -> {
                    logger.warn("bidRetargeting $retId: unexpected number of duplicates, ${retObjects.size}")
                }
            }
        }
        return deleted
    }

    private fun readRetargetingBidsDuplicates(retIds: Collection<Long>): Map<Long, List<DedupObject>> {
        val shards = shardHelper.dbShards()

        val idsToShard = mutableMapOf<Long, MutableList<DedupObject>>()
        for (shard in shards) {
            val bidRetargetingObjects = dslContextProvider.ppc(shard)
                .select(BIDS_RETARGETING.RET_ID)
                .select(BIDS_RETARGETING.STATUS_BS_SYNCED)
                .from(BIDS_RETARGETING)
                .where(BIDS_RETARGETING.RET_ID.`in`(retIds))
                .fetch { mapBidRetargeting(it, shard) }
            bidRetargetingObjects.forEach { idsToShard.getOrPut(it.id) { ArrayList(2) }.add(it) }
        }
        return idsToShard
    }

    private fun deleteBidRetargetingDuplicate(obj: DedupObject) {
        dslContextProvider.ppc(obj.shard)
            .deleteFrom(BidsRetargeting.BIDS_RETARGETING)
            .where(BidsRetargeting.BIDS_RETARGETING.RET_ID.eq(obj.id))
            .execute()
        essClient.addLogicObjectsForProcessor(
            obj.shard, logicProcessorName,
            listOf(
                Mysql2GrutReplicationObject(
                    biddableShowConditionId = obj.id,
                    biddableShowConditionType = BiddableShowConditionChangeType.RETARGETING,
                    isDeleted = true,
                )
            )
        )
        logger.info("Deleted bidRetargeting from MySQL: id ${obj.id}, shard ${obj.shard}")
    }

    private fun mapBidRetargeting(record: Record, shard: Int): DedupObject {
        return DedupObject(
            record.get(BIDS_RETARGETING.RET_ID),
            shard,
            StatusBsSynced.valueOfDbFormat(record.get(BIDS_RETARGETING.STATUS_BS_SYNCED).literal),
        )
    }

    private fun readIdsToDedup(
        ytCluster: YtCluster,
        tablePath: String,
        startRow: Long,
        lastRow: Long
    ): List<DedupObjectIds> {
        val objectIds = mutableListOf<DedupObjectIds>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(YtTable(tablePath), {
                objectIds.add(
                    DedupObjectIds(
                        retargetingBidId = it.retargetingBidId,
                        shard = it.shard,
                    )
                )
            }, DedupTableRow(), startRow, lastRow)
        return objectIds
    }
}

class DedupTableRow : YtTableRow(ALL_COLUMNS) {
    companion object {
        val RETARGETING_BID_ID = YtField("retargeting_bid_id", Long::class.java)
        val SHARD = YtField("shard", Int::class.java)

        val ALL_COLUMNS = listOf(RETARGETING_BID_ID)
    }

    val retargetingBidId: Long?
        get() = valueOf(RETARGETING_BID_ID)

    val shard: Int
        get() = valueOf(SHARD)
}

data class DedupObjectIds(
    val retargetingBidId: Long? = null,
    val shard: Int,
)

data class DedupObject(val id: Long, val shard: Int, val statusBsSynced: StatusBsSynced)
