package ru.yandex.direct.oneshot.oneshots.mysql2grut.delete_objects

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.container.AdGroupsSelectionCriteria
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.SupportedCampaignsService
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.oneshot.oneshots.mysql2grut.Mysql2GrutYtUtils
import ru.yandex.direct.oneshot.oneshots.mysql2grut.Param
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class State(
    val lastRow: Long
)

data class ObjectIds(
    val clientId: Long? = null,
    val campaignId: Long? = null,
    val adGroupId: Long? = null,
    val bannerId: Long? = null,
    val retCondId: Long? = null,
)

class InputTableRow : YtTableRow(ALL_COLUMNS) {
    companion object {
        val CLIENT_ID = YtField("client_id", Long::class.java)
        val CAMPAIGN_ID = YtField("campaign_id", Long::class.java)
        val ADGROUP_ID = YtField("adgroup_id", Long::class.java)
        val BANNER_ID = YtField("banner_id", Long::class.java)
        val RET_COND_ID = YtField("retargeting_condition_id", Long::class.java)

        val ALL_COLUMNS = listOf(CLIENT_ID, CAMPAIGN_ID, ADGROUP_ID, BANNER_ID, RET_COND_ID)
    }

    val clientId: Long?
        get() = valueOf(CLIENT_ID)

    val campaignId: Long?
        get() = valueOf(CAMPAIGN_ID)

    val adGroupId: Long?
        get() = valueOf(ADGROUP_ID)

    val bannerId: Long?
        get() = valueOf(BANNER_ID)

    val retCondId: Long?
        get() = valueOf(RET_COND_ID)
}

/**
 * Ваншот для удаления данных из GRuT, которые приехали в результате репликации, а их удаления после не приехали

 * Для запуска ваншота нужно подготовить таблицу на yt с колонкой из ALL_COLUMNS
 * Передать таблицу в параметрах ваншота `{"tablePath": "//home/direct/test/mspirit/clients_for_resync", "ytCluster": "HAHN"}`
 */
@Component
@Approvers("mspirit", "ppalex", "elwood")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class DeleteObjectsFromGrutOneshot(
    private val ytProvider: YtProvider,
    private val shardHelper: ShardHelper,
    private val grutApiService: GrutApiService,
    private val supportedCampaignsService: SupportedCampaignsService,
    private val adGroupRepository: AdGroupRepository,
    private val bannerRepository: BannerTypedRepository,
    private val retargetingConditionRepository: RetargetingConditionRepository,
) : SimpleOneshot<Param, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(DeleteObjectsFromGrutOneshot::class.java)
    }

    private val ytUtils = Mysql2GrutYtUtils(ytProvider)

    override fun validate(inputData: Param) = ytUtils.validate(inputData)

    override fun execute(inputData: Param, prevState: State?): State? {
        val chunkSize = 10000
        val relaxTime = 10L
        logger.info("Start from state=$prevState with chunkSize=$chunkSize, relaxTime = $relaxTime")

        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + chunkSize
        val objectIds = getIdsToSend(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        val deletedClients = deleteClients(objectIds)
        val deletedCampaigns = deleteCampaigns(objectIds)
        val deletedAdGroups = deleteAdGroups(objectIds)
        val deletedBanners = deleteBanners(objectIds)
        val deletedRetConds = deleteRetargetingConditions(objectIds)

        val deletedObjects = deletedClients + deletedCampaigns + deletedAdGroups + deletedBanners + deletedRetConds
        if (deletedObjects != 0) {
            logger.info("Iteration finished, sleep for $relaxTime seconds")
            Thread.sleep(relaxTime * 1000)
        } else {
            logger.info("No handled objects on iteration")
        }
        return if (objectIds.size < chunkSize) null else State(lastRow)
    }

    private fun deleteClients(objectIdsList: List<ObjectIds>): Int {
        val clientsIds = objectIdsList
            .mapNotNull { it.clientId }
            .distinct()
        if (clientsIds.isEmpty()) {
            return 0
        }
        val existingClients = shardHelper.groupByShard(clientsIds, ShardKey.CLIENT_ID)
            .shardedDataMap.flatMap { it.value }

        if (existingClients.isNotEmpty()) {
            logger.info("Can't delete ${existingClients.size} clients because exising in mysql: $existingClients")
        }
        val clientsToDelete = clientsIds.minus(existingClients)
        logger.info("Going to delete ${clientsToDelete.size} clients")
        grutApiService.clientGrutDao.deleteObjects(clientsToDelete)
        return clientsToDelete.size
    }

    private fun deleteCampaigns(objectIdsList: List<ObjectIds>): Int {

        val campaignsIds = objectIdsList
            .mapNotNull { it.campaignId }
            .distinct()
        if (campaignsIds.isEmpty()) {
            return 0
        }
        val existingCampaigns = shardHelper.groupByShard(campaignsIds, ShardKey.CID)
            .shardedDataMap
            .flatMap {
                supportedCampaignsService.getSafely(it.key, it.value, CommonCampaign::class.java)
                    .filter { camp -> camp.statusEmpty == false }.map { camp -> camp.id }

            }.toSet()

        if (existingCampaigns.isNotEmpty()) {
            logger.info("Can't delete ${existingCampaigns.size} campaigns because exising in mysql: $existingCampaigns")
        }
        val campaignsToDelete = campaignsIds.minus(existingCampaigns)
        logger.info("Going to delete ${campaignsToDelete.size} campaigns")
        grutApiService.campaignGrutDao.deleteCampaignsByDirectIds(campaignsToDelete)

        return campaignsToDelete.size
    }

    private fun deleteAdGroups(objectIdsList: List<ObjectIds>): Int {
        val adGroupIds = objectIdsList
            .mapNotNull { it.adGroupId }
            .toSet()
        if (adGroupIds.isEmpty()) {
            return 0
        }
        val existingAdGroups = shardHelper.groupByShard(adGroupIds, ShardKey.PID)
            .shardedDataMap
            .flatMap {
                val selectionCriteria = AdGroupsSelectionCriteria().withAdGroupIds(it.value.toSet())
                adGroupRepository.getAdGroupIdsBySelectionCriteria(it.key, selectionCriteria, LimitOffset.maxLimited())
            }.toSet()
        if (existingAdGroups.isNotEmpty()) {
            logger.info("Can't delete ${existingAdGroups.size} adGroups because exising in mysql: $existingAdGroups")
        }
        val adGroupToDelete = adGroupIds.minus(existingAdGroups)
        logger.info("Going to delete ${adGroupToDelete.size} adGroups")
        grutApiService.adGroupGrutDao.deleteObjects(adGroupToDelete)
        return adGroupToDelete.size
    }

    private fun deleteBanners(objectIdsList: List<ObjectIds>): Int {
        val bannerIds = objectIdsList
            .mapNotNull { it.bannerId }
            .toSet()
        if (bannerIds.isEmpty()) {
            return 0
        }
        val existingBanners = shardHelper.groupByShard(bannerIds, ShardKey.BID)
            .shardedDataMap
            .flatMap {
                bannerRepository.getSafely(it.key, it.value, BannerWithSystemFields::class.java).map { it.id }
            }.toSet()
        if (existingBanners.isNotEmpty()) {
            logger.info("Can't delete ${existingBanners.size} campaigns because exising in mysql: $existingBanners")
        }
        val bannersToDelete = bannerIds.minus(existingBanners)
        logger.info("Going to delete ${bannersToDelete.size} banners")
        grutApiService.bannerReplicationApiService.deleteObjects(bannersToDelete)

        return bannersToDelete.size
    }

    private fun deleteRetargetingConditions(objectIdsList: List<ObjectIds>): Int {
        val retCondIds = objectIdsList
            .mapNotNull { it.retCondId }
            .toSet()
        if (retCondIds.isEmpty()) {
            return 0
        }

        val retCondsByShard = shardHelper.groupByShard(retCondIds, ShardKey.RET_COND_ID).shardedDataMap
        val existing = mutableSetOf<Long>()
        for ((shard, retConds) in retCondsByShard) {
            val retCondsIds = retargetingConditionRepository.getConditions(shard, retConds).map { it.id }
            existing.addAll(retCondsIds)
        }

        if (existing.isNotEmpty()) {
            logger.info("Can't delete ${existing.size} retargeting_conditions because exising in mysql: $existing")
        }
        val toDelete = retCondIds.minus(existing.map { it }.toSet())
        if (toDelete.isNotEmpty()) {
            logger.info("Going to delete ${toDelete.size} retargeting_conditions")
            grutApiService.retargetingConditionGrutApi.deleteObjects(toDelete)
        } else {
            logger.info("All objects still exist in MySQL on current iteration")
        }
        return toDelete.size
    }

    private fun getIdsToSend(ytCluster: YtCluster, tablePath: String, startRow: Long, lastRow: Long): List<ObjectIds> {
        val objectIds = mutableListOf<ObjectIds>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(YtTable(tablePath), {
                objectIds.add(
                    ObjectIds(
                        clientId = it.clientId,
                        campaignId = it.campaignId,
                        adGroupId = it.adGroupId,
                        bannerId = it.bannerId,
                    )
                )
            }, InputTableRow(), startRow, lastRow)
        return objectIds
    }
}
