package ru.yandex.direct.oneshot.oneshots.offeracceptedinit

import org.jooq.impl.DSL.count
import org.slf4j.LoggerFactory.getLogger
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.Tables.*
import ru.yandex.direct.dbschema.ppc.enums.ClientsRole.client
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsIsOfferAccepted
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class State(
        val lastClientId: ClientId
)

@Component
@Approvers("maxlog", "adubinkin")
class OfferAcceptedInitOneshot @Autowired constructor(
        private val dsl: DslContextProvider
) : ShardedOneshot<Void?, State?> {
    companion object {
        private val logger = getLogger(OfferAcceptedInitOneshot::class.java)
        private const val ONE_STEP_LIMIT = 10_000L
    }

    override fun validate(inputData: Void?): ValidationResult<Void?, Defect<Any>>? = ValidationResult.success(null)

    override fun execute(inputData: Void?, prevState: State?, shard: Int): State? {
        val prevClientId = prevState?.lastClientId ?: ClientId.fromLong(0L)
        val (lastClientId, uids) = getUidsForChanging(shard, prevClientId)
        if (lastClientId == null) {
            logger.info("There are no rows for changing, shard {}", shard)
            return null
        }
        logger.info("Migrating {} rows to new value, shard {}", uids.size, shard)
        val rowsNumber = updateOfferAcceptedField(shard, uids)
        logger.info("The number of updated records: {}, shard {}", rowsNumber, shard)
        return State(lastClientId)
    }

    private fun getUidsForChanging(shard: Int, prevClientId: ClientId): Pair<ClientId?, List<Long>> {
        val clientIds = dsl.ppc(shard)
                .select(CLIENTS.CLIENT_ID)
                .from(CLIENTS)
                .innerJoin(USERS).on(USERS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(
                        CLIENTS.CLIENT_ID.greaterThan(prevClientId.asLong())
                                .and(CLIENTS.AGENCY_CLIENT_ID.isNull)
                                .and(CLIENTS.ROLE.eq(client))
                                .and(CLIENTS.DELETED_REPS.isNull.or(CLIENTS.DELETED_REPS.notContains("{")))
                )
                .groupBy(CLIENTS.CLIENT_ID)
                .having(count().eq(1))
                .orderBy(CLIENTS.CLIENT_ID.asc())
                .limit(ONE_STEP_LIMIT)
                .fetch(CLIENTS.CLIENT_ID)
        val uids = dsl.ppc(shard)
                .select(USERS.UID)
                .from(USERS)
                .where(
                        USERS.CLIENT_ID.`in`(clientIds)
                )
                .fetch(USERS.UID)
        val lastClientId = ClientId.fromNullableLong(clientIds.lastOrNull())
        return Pair(lastClientId, uids)
    }

    private fun updateOfferAcceptedField(shard: Int, uids: List<Long>) =
            dsl.ppc(shard)
                    .update(USERS_OPTIONS)
                    .set(USERS_OPTIONS.IS_OFFER_ACCEPTED, UsersOptionsIsOfferAccepted.Yes)
                    .where(USERS_OPTIONS.UID.`in`(uids))
                    .execute()

}
