package ru.yandex.direct.oneshot.oneshots.offlinereports

import org.jooq.impl.DSL.trueCondition
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileCustomName
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType
import ru.yandex.direct.core.entity.mdsfile.repository.MdsFileRepository
import ru.yandex.direct.dbschema.ppc.tables.MdsMetadata.MDS_METADATA
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SafeOneshot
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class AddMdsCustomNameForXlsReportsInputData(
    val mdsIds: List<Long>?
)

@Component
@SafeOneshot
@Multilaunch
class AddMdsCustomNameForXlsReportsOneshot @Autowired constructor(
    private val mdsFileRepository: MdsFileRepository,
    private val dslContextProvider: DslContextProvider,
) : ShardedOneshot<AddMdsCustomNameForXlsReportsInputData, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(AddMdsCustomNameForXlsReportsOneshot::class.java);
        private const val chunk_size = 50;
    }

    override fun validate(inputData: AddMdsCustomNameForXlsReportsInputData?): ValidationResult<AddMdsCustomNameForXlsReportsInputData, Defect<Any>> =
        ValidationResult.success(inputData)

    override fun execute(inputData: AddMdsCustomNameForXlsReportsInputData?, prevState: Void?, shard: Int): Void? {
        var borderMdsId: Long = 0;
        val requestedMdsIds = inputData?.mdsIds;
        while (true) {
            if (requestedMdsIds != null && (requestedMdsIds.maxOrNull() ?: 0) <= borderMdsId) {
                break
            }
            val mdsFilenameForId = dslContextProvider.ppc(shard).select(MDS_METADATA.ID, MDS_METADATA.FILENAME)
                .from(MDS_METADATA)
                .where(MDS_METADATA.TYPE.eq(MdsStorageType.toSource(MdsStorageType.XLS_REPORTS)))
                .and(
                    if (requestedMdsIds != null) {
                        MDS_METADATA.ID.`in`(requestedMdsIds)
                    } else {
                        trueCondition()
                    }
                )
                .and(MDS_METADATA.ID.gt(borderMdsId))
                .limit(chunk_size)
                .fetchMap(MDS_METADATA.ID, MDS_METADATA.FILENAME);
            val mdsIds = mdsFilenameForId.keys.toList();
            borderMdsId = mdsIds.maxOrNull() ?: break
            val mdsCustomNames = mdsFileRepository.getCustomNames(shard, mdsIds);
            val mdsIdsWithoutCustomNames = mdsIds.filter { !mdsCustomNames.containsKey(it) };
            logger.info("mds ids without custom names: {}", mdsIdsWithoutCustomNames);
            val newMdsCustomNames = mdsIdsWithoutCustomNames.map {
                MdsFileCustomName().withMdsId(it).withFilename(mdsFilenameForId[it])
            }
            val newCustomNameIds = mdsFileRepository.addCustomName(shard, newMdsCustomNames);
            logger.info("inserted {} records into mds_custom_names", newCustomNameIds.size)
        }
        return null
    }
}
