package ru.yandex.direct.oneshot.oneshots.offlinereports

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.offlinereport.repository.OfflineReportRepository
import ru.yandex.direct.dbschema.ppc.tables.OfflineReports.OFFLINE_REPORTS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeField
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeObject
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes
import java.net.HttpURLConnection
import java.net.URL
import java.util.function.Consumer

data class InputData(
    val ytCluster: YtCluster,
    val tablePath: String
)

data class State(
    val lastRow: Long = 0L
)

@YTreeObject
data class InputTableRow(
    @YTreeField(key = "offline_report_id") val offlineReportId: Long?
)

/**
 * Ваншот для удаления нерабочих ссылок на оффлайн-отчёты (удалялись по ошибке, решили не восстанавливать https://st.yandex-team.ru/DIRECT-146121#60b8c0db07aeb74cb7e583e0)
 *  На вход принимает таблицу в YT из одного столбца с id отчёта в таблице offline_reports.
 *  Дополнительно проверяет, что url для отчёта действительно нерабочий (отдаёт 404), в противном случае отчёт не удаляет.
 */
@Approvers("yukaba", "dlyange")
@Multilaunch
@Component
class RemoveDeletedOfflineReportLinksOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val dslContextProvider: DslContextProvider,
    private val offlineReportRepository: OfflineReportRepository,
) : ShardedOneshot<InputData, State> {
    companion object {
        private val logger = LoggerFactory.getLogger(RemoveDeletedOfflineReportLinksOneshot::class.java);
        private const val chunkSize = 100;
    }

    override fun validate(inputData: InputData): ValidationResult<InputData, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)

        vb.item(inputData.ytCluster, "ytCluster").check(notNull())

        if (vb.result.hasAnyErrors()) return vb.result

        vb.item(inputData.tablePath, "tablePath")
            .check(notNull())
            .check(Constraint.fromPredicate(
                { tableName -> ytProvider.getOperator(inputData.ytCluster).exists(YtTable(tableName)) },
                objectNotFound()))
        return vb.result
    }

    private fun readInputTable(inputData: InputData, startRow: Long, lastRow: Long): List<Long> {
        val entryType = YTableEntryTypes.yson(InputTableRow::class.java)
        val ytTable = YtTable(inputData.tablePath)

        val items = mutableListOf<Long>()
        ytProvider.get(inputData.ytCluster).tables()
            .read(ytTable.ypath().withRange(startRow, lastRow), entryType,
                Consumer { row -> items.add(row.offlineReportId!!) })
        return items
    }

    override fun execute(inputData: InputData, prevState: State?, shard: Int): State? {
        val startRow = prevState?.lastRow ?: 0L
        val lastRow = startRow + chunkSize
        logger.info("shard=$shard: Start from row=$startRow, to row=$lastRow (excluded)")

        val reportIds = readInputTable(inputData, startRow, lastRow)

        for (reportId in reportIds) {
            val report = offlineReportRepository.getOfflineReport(shard, reportId) ?: continue
            val reportUrl = report.reportUrl
            logger.info("shard={}, report_id={}: checking url: {}", shard, reportId, reportUrl)
            val url = URL(reportUrl);
            val con = url.openConnection() as HttpURLConnection;
            con.requestMethod = "HEAD"
            if (con.responseCode == 404) {
                logger.info("shard={}, report_id={}: url is unavailable, deleting report: {}", shard, reportId,report)
                dslContextProvider.ppc(shard).deleteFrom(OFFLINE_REPORTS).where(OFFLINE_REPORTS.OFFLINE_REPORT_ID.eq(reportId)).execute()
                Thread.sleep(1000);
            } else {
                logger.info("shard={}, report_id={}: url is available, NOT deleting", shard, reportId, reportUrl)
            }
        }
        return if (reportIds.size < chunkSize) {
            logger.info("shard={}: last iteration finished", shard)
            null
        } else {
            State(lastRow);
        }
    }
}
