package ru.yandex.direct.oneshot.oneshots.package_strategy_migration

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.CAMPAIGN_TO_PACKAGE_STRATEGY_MIGRATION_CHUNK_SIZE
import ru.yandex.direct.common.db.PpcPropertyNames.CAMPAIGN_TO_PACKAGE_STRATEGY_MIGRATION_RELAX_TIME
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.base.YtState
import ru.yandex.direct.oneshot.oneshots.package_strategy_migration.CampaignToPackageStrategyMigrationOneShot.Companion.InputData
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtOperator
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import kotlin.math.min

/*
* Шардированный ваншот мигрирующий кампании в пакетные стратегии.
* На вход принимает табличку в YT со списком кампаний.
* Ваншот имеет 2 режима работы:
*   - сохранение пакетной стратегии с валидацией
*   - сохранение без валидации.
* 
* ВАЖНО! Ваншот перезатирает стратегии.
* 
* Для более эффективной работы ваншота стоит группировать в YT таблицы идентификаторы кампаний
* по идентификатору клиента.
* */
@Component
@Approvers("ssdmitriev", "pavryabov", "ruslansd", "ninazhevtyak", "kuvshinov")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class CampaignToPackageStrategyMigrationOneShot(
    private val ytProvider: YtProvider,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val dslContextProvider: DslContextProvider,
    private val strategyMigrationService: StrategyMigrationService,
    private val ppcPropertiesSupport: PpcPropertiesSupport
) : ShardedOneshot<InputData, YtState> {

    private val logger = LoggerFactory.getLogger(CampaignToPackageStrategyMigrationOneShot::class.java)

    companion object {
        private const val DEFAULT_CHUNK_SIZE = 1000L

        class PackageStrategyMigrationRow : YtTableRow(listOf(CID)) {
            companion object {
                private val CID = YtField("cid", Long::class.java)
            }

            val cid: Long get() = valueOf(CID).toLong()
        }

        data class InputData(
            override val ytCluster: String,
            override val tablePath: String,
            val operatorUid: Long,
            val saveWithValidation: Boolean?
        ) : BaseInputData
    }

    override fun validate(inputData: InputData): ValidationResult<InputData, Defect<*>> {
        fun tableExists(tablePath: String): Boolean {
            val cluster = YtCluster.parse(inputData.ytCluster.lowercase())
            return ytProvider.getOperator(cluster).exists(YtTable(tablePath))
        }

        val validator = InputDataValidator<InputData> { tableExists(it) }
        return validator.apply(inputData)
    }

    override fun execute(inputData: InputData, prevState: YtState?, shard: Int): YtState? {
        val ytCluster = YtCluster.parse(inputData.ytCluster.lowercase())
        val ytTable = YtTable(inputData.tablePath)
        val ytOperator: YtOperator = ytProvider.getOperator(ytCluster)
        val chunkSize =
            ppcPropertiesSupport.get(CAMPAIGN_TO_PACKAGE_STRATEGY_MIGRATION_CHUNK_SIZE).getOrDefault(DEFAULT_CHUNK_SIZE)
        val currentState = prevState ?: firstIteration(shard, ytOperator.readTableRowCount(ytTable))
        return process(
            currentState,
            ytOperator,
            ytTable,
            inputData.operatorUid,
            inputData.saveWithValidation ?: false,
            chunkSize,
            shard
        )
    }

    private fun process(
        state: YtState,
        ytOperator: YtOperator,
        ytTable: YtTable,
        operatorUid: Long,
        saveWithValidation: Boolean,
        chunkSize: Long,
        shard: Int
    ): YtState? {
        val rowCount: Long = state.totalRowCount
        val startRow: Long = state.nextRow
        val endRow = min(startRow + chunkSize, rowCount)
        if (startRow >= rowCount) {
            logger.info("Last iteration, last processed row: {}, total rows: {}", startRow, rowCount)
            return null
        }
        val parsedRows = mutableListOf<Long>()
        ytOperator.readTableByRowRange(
            ytTable,
            { parsedRows.add(it.cid) },
            PackageStrategyMigrationRow(),
            startRow,
            endRow
        )
        logger.info("Ready to process ${parsedRows.size} rows, from [$startRow] to [$endRow]")
        processChunk(shard, parsedRows.toSet(), operatorUid, saveWithValidation)
        logger.info("Processed ${parsedRows.size} rows, from [$startRow] to [$endRow]")
        return YtState()
            .withNextRowFromYtTable(endRow)
            .withTotalRowCount(rowCount)
    }

    private fun firstIteration(shard: Int, totalRow: Long): YtState {
        logger.info("First iteration for shard [$shard]")
        return YtState()
            .withNextRowFromYtTable(0L)
            .withTotalRowCount(totalRow)
    }

    private fun processChunk(
        shard: Int,
        campaignIds: Set<Long>,
        operatorUid: Long,
        saveWithValidation: Boolean
    ) {
        val campaigns = campaignTypedRepository.getTyped(dslContextProvider.ppc(shard), campaignIds)
        if (campaigns.isNotEmpty()) {
            val relaxTimeInSeconds =
                ppcPropertiesSupport.get(CAMPAIGN_TO_PACKAGE_STRATEGY_MIGRATION_RELAX_TIME).getOrDefault(5L)
            val statistics = strategyMigrationService.migrate(shard, campaigns, saveWithValidation, operatorUid)
            logger.info("Chunk processed in saveWithValidation=$saveWithValidation mode with statistics [$statistics]")
            logger.info("Sleep for [$relaxTimeInSeconds] seconds")
            Thread.sleep(relaxTimeInSeconds * 1000L)
        }
    }
}
