package ru.yandex.direct.oneshot.oneshots.package_strategy_migration

import org.jooq.DSLContext
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.PACKAGE_STRATEGY_IDENTIFIERS_MIGRATION_BATCH_LIMIT
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.dbschema.ppc.tables.Strategies
import ru.yandex.direct.dbschema.ppcdict.Tables
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.oneshots.CampaignMigrationBaseOneshot
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail

@Component
@Multilaunch
@PausedStatusOnFail
@Approvers("ruslansd", "pavryabov", "ssdmitriev")
class PackageStrategyIdentifierOneShot(
    campaignTypedRepository: CampaignTypedRepository,
    private val campaignModifyRepository: CampaignModifyRepository,
    private val dslContextProvider: DslContextProvider,
    private val strategyPpcDictDeleteRepository: StrategyPpcDictDeleteRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport
) : CampaignMigrationBaseOneshot<Unit>(campaignTypedRepository) {

    private val batchSizeProperty = ppcPropertiesSupport.get(PACKAGE_STRATEGY_IDENTIFIERS_MIGRATION_BATCH_LIMIT)

    override fun process(shard: Int, campaigns: List<BaseCampaign>) {
        val campaignsToUpdate =
            campaigns.mapNotNull { it as? CampaignWithPackageStrategy }
                .filter { needToUpdateStrategyId(it) }

        val campaignAppliedChanges = campaignsToUpdate.map(this::campaignChanges)
        if (campaignAppliedChanges.isNotEmpty()) {
            val strategyIdUnits = campaignAppliedChanges.map { StrategyIdUnit(it.model.strategyId, it.model.clientId) }
            //Добавляем новые идентификаторы в shard_inc_strategy_id
            addStrategyIdsToPpcDict(strategyIdUnits)

            //В рамках транзакции производим апдейты
            dslContextProvider.ppcTransaction(shard) { configuration ->
                val dsl = configuration.dsl()
                update(dsl, campaignAppliedChanges)
            }
            // Удаляем старые идентификаторы из shard_inc_strategy_id
            strategyPpcDictDeleteRepository.delete(
                dslContextProvider.ppcdict(),
                campaignAppliedChanges.mapNotNull { it.getOldValue(CampaignWithPackageStrategy.STRATEGY_ID) }
            )
        }
    }

    override fun classes(): Set<Class<out BaseCampaign>> = setOf(CampaignWithPackageStrategy::class.java)

    override fun batchLimit(): Int {
        return batchSizeProperty.getOrDefault(2000)
    }

    private fun update(
        dslContext: DSLContext,
        campaignAppliedChanges: List<AppliedChanges<CampaignWithPackageStrategy>>
    ) {
        val strategyUpdateUnits = campaignAppliedChanges.map(this::strategyIdUpdateUnits)
        //Обновляем значение `campaigns.strategy_id`
        campaignModifyRepository.updateCampaignsTable(dslContext, campaignAppliedChanges)
        //Обновляем strategies.strategy_id
        updateStrategyId(dslContext, strategyUpdateUnits)
    }

    private fun campaignChanges(campaign: CampaignWithPackageStrategy): AppliedChanges<CampaignWithPackageStrategy> {
        return ModelChanges(campaign.id, CampaignWithPackageStrategy::class.java)
            .process(campaign.orderId, CampaignWithPackageStrategy.STRATEGY_ID)
            .applyTo(campaign)
    }

    private fun strategyIdUpdateUnits(appliedChanges: AppliedChanges<CampaignWithPackageStrategy>): StrategyIdUpdateUnit {
        return StrategyIdUpdateUnit(
            appliedChanges.getOldValue(CampaignWithPackageStrategy.STRATEGY_ID)!!,
            appliedChanges.model.strategyId
        )
    }

    private fun updateStrategyId(
        dslContext: DSLContext,
        units: List<StrategyIdUpdateUnit>
    ) {
        val updateQueries = units.map { unit ->
            dslContext.update(Strategies.STRATEGIES)
                .set(Strategies.STRATEGIES.STRATEGY_ID, unit.new)
                .where(Strategies.STRATEGIES.STRATEGY_ID.eq(unit.old))
        }
        dslContext.batch(updateQueries).execute()
    }

    private fun addStrategyIdsToPpcDict(units: List<StrategyIdUnit>) {
        val context = dslContextProvider.ppcdict()
        val insertStep = context.insertInto(Tables.SHARD_INC_STRATEGY_ID)
            .columns(Tables.SHARD_INC_STRATEGY_ID.STRATEGY_ID, Tables.SHARD_INC_STRATEGY_ID.CLIENT_ID)

        units.forEach { unit ->
            insertStep.values(unit.strategyId, unit.clientId)
        }
        insertStep.onDuplicateKeyIgnore()
            .execute()
    }

    private fun needToUpdateStrategyId(campaign: CampaignWithPackageStrategy): Boolean =
        campaign.orderId ?: 0L != 0L &&
            campaign.strategyId ?: 0L != 0L &&
            campaign.orderId != campaign.strategyId

    internal data class StrategyIdUpdateUnit(val old: Long, val new: Long)
    internal data class StrategyIdUnit(val strategyId: Long, val clientId: Long)
}
