package ru.yandex.direct.oneshot.oneshots.package_strategy_migration

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository
import ru.yandex.direct.core.entity.strategy.container.StrategyOperationOptions
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.StrategyName
import ru.yandex.direct.core.entity.strategy.repository.StrategyModifyRepository
import ru.yandex.direct.core.entity.strategy.service.StrategyOperationFactory
import ru.yandex.direct.core.entity.strategy.service.converter.CampaignToStrategyConverterService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.ModelChanges
import java.time.LocalDateTime

@Component
class StrategyMigrationService(
    private val dslContextProvider: DslContextProvider,
    private val strategyOperationFactory: StrategyOperationFactory,
    private val strategyModifyRepository: StrategyModifyRepository,
    private val campaignModifyRepository: CampaignModifyRepository,
    private val converterService: CampaignToStrategyConverterService,
    private val strategyCommonDeleteRepository: StrategyCommonDeleteRepository,
    private val strategyPpcDictDeleteRepository: StrategyPpcDictDeleteRepository
) {

    private val logger = LoggerFactory.getLogger(StrategyMigrationService::class.java)

    fun migrate(
        shard: Int,
        campaigns: List<BaseCampaign>,
        saveWithValidation: Boolean,
        operatorUid: Long
    ): Statistics {
        val now = LocalDateTime.now()
        val availableToConvertCampaigns = campaigns
            .mapNotNull { it as? CampaignWithPackageStrategy }
            .filter { isCampaignSupportedToConvert(it) }
        val statistics = Statistics(
            shard = shard,
            campaignToProcessCount = campaigns.size,
            campaignAvailableToConvertCount = availableToConvertCampaigns.size
        )
        val strategies = availableToConvertCampaigns.map { campaign ->
            val strategy = converterService.toStrategyWithIdEqualToOrderId(
                ClientId.fromLong(campaign.clientId),
                campaign.lastChange ?: now,
                campaign
            )
            CampaignAndStrategy(campaign, strategy)
        }
        deleteAlreadySavedStrategies(shard, strategies)
        dropStrategyIdInCampaigns(shard, campaigns)
        statistics.strategyCount = strategies.size
        statistics.savedStrategyCount = save(shard, strategies, saveWithValidation, operatorUid)
        return statistics
    }

    private fun deleteAlreadySavedStrategies(shard: Int, strategies: List<CampaignAndStrategy>) {
        //пытаемся удалить все записи батча DIRECT-165027
        val maybeSavedStrategyIds = strategies
            .map { it.strategy.id }
            .filter { strategyId -> strategyId != 0L }

        deleteAlreadySavedStrategiesByIds(shard, maybeSavedStrategyIds)
        strategyPpcDictDeleteRepository.delete(dslContextProvider.ppcdict(), maybeSavedStrategyIds)
    }

    private fun dropStrategyIdInCampaigns(shard: Int, campaigns: List<BaseCampaign>) {
        val changes = campaigns.mapNotNull { campaign ->
            if (campaign is CampaignWithPackageStrategy && campaign.strategyId != 0L) {
                ModelChanges(campaign.id, CampaignWithPackageStrategy::class.java)
                    .process(0L, CampaignWithPackageStrategy.STRATEGY_ID)
                    .applyTo(campaign)
            } else {
                null
            }
        }
        if (changes.isNotEmpty()) {
            campaignModifyRepository.updateCampaignsTable(
                dslContextProvider.ppc(shard),
                changes
            )
        }
    }

    private fun deleteAlreadySavedStrategiesByIds(shard: Int, strategyIds: List<Long>) {
        val deleteOperation = StrategyDeleteOperation(
            strategyIds,
            shard,
            dslContextProvider,
            strategyCommonDeleteRepository
        )
        deleteOperation.prepareAndApply()
    }

    private fun save(
        shard: Int,
        strategies: List<CampaignAndStrategy>,
        saveWithValidation: Boolean,
        operatorUid: Long
    ): Int {
        var savedStrategyCount = 0
        if (saveWithValidation) {
            savedStrategyCount += saveWithValidation(shard, strategies, operatorUid)
        } else {
            saveWithoutValidation(shard, strategies)
            savedStrategyCount += strategies.size
        }
        return savedStrategyCount
    }

    private fun saveWithValidation(
        shard: Int,
        strategies: List<CampaignAndStrategy>,
        operatorUid: Long
    ): Int {
        val addOperations = strategies.groupBy { campaignAndStrategy -> campaignAndStrategy.campaign.clientId }
            .map { (clientId, campaignAndStrategies) ->
                strategyOperationFactory.createStrategyAddOperation(
                    shard,
                    operatorUid,
                    ClientId.fromLong(clientId),
                    operatorUid,
                    campaignAndStrategies.map(CampaignAndStrategy::strategy),
                    StrategyOperationOptions(
                        isStrategyIdsAlreadyFilled = true,
                        isCampaignToPackageStrategyOneshot = true
                    )
                )
            }
        logger.info("Prepared [${addOperations.size}] operations")
        var savedStrategyCount = 0
        addOperations.forEach {
            val massResult = it.prepareAndApply()
            savedStrategyCount += massResult.successfulCount
            if (massResult.errorCount > 0) {
                val flattenErrorsByCid = it.models.zip(massResult.result)
                    .filter { (_, result) -> !result.isSuccessful }
                    .associate { (model, result) -> model.id to result.validationResult.flattenErrors() }
                logger.info("There are validation errors [{}]", flattenErrorsByCid)
            }
        }
        return savedStrategyCount
    }

    private fun saveWithoutValidation(
        shard: Int,
        strategies: List<CampaignAndStrategy>
    ) {
        strategies.groupBy { campaignAndStrategy -> campaignAndStrategy.campaign.clientId }
            .forEach { (client, campaignAndStrategies) ->
                val container = StrategyRepositoryContainer(
                    shard,
                    ClientId.fromLong(client),
                    emptyMap(),
                    true
                )
                dslContextProvider.ppcTransaction(shard) { conf ->
                    val dslContext = conf.dsl()
                    strategyModifyRepository.add(
                        dslContext,
                        container,
                        campaignAndStrategies.map(CampaignAndStrategy::strategy)
                    )
                    val changes = campaignAndStrategies.map { campaignAndStrategy ->
                        ModelChanges(campaignAndStrategy.campaign.id, CampaignWithPackageStrategy::class.java)
                            .process(campaignAndStrategy.strategy.id, CampaignWithPackageStrategy.STRATEGY_ID)
                            .applyTo(campaignAndStrategy.campaign)
                    }
                    campaignModifyRepository.updateCampaignsTable(
                        dslContext,
                        changes
                    )
                }
            }
    }

    companion object {
        data class CampaignAndStrategy(
            val campaign: CampaignWithPackageStrategy,
            val strategy: BaseStrategy
        )

        data class Statistics(
            val shard: Int,
            var campaignToProcessCount: Int = 0,
            var campaignAvailableToConvertCount: Int = 0,
            var strategyCount: Int = 0,
            var savedStrategyCount: Int = 0
        )

        private val supportedStrategyNames = StrategyName.values().map { it.name }.toSet()

        private fun isCampaignSupportedToConvert(campaign: CampaignWithPackageStrategy): Boolean {
            return supportedStrategyNames.contains(campaign.strategy.strategyName.name)
        }
    }
}
