@file:Suppress("unused")

package ru.yandex.direct.oneshot.oneshots.performancefilter

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign
import ru.yandex.direct.core.entity.campaign.model.SmartCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.dynamictextadtarget.container.DynamicTextAdTargetSelectionCriteria
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTargetState
import ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicTextAdTargetService
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toDynamicFilters
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toSmartFilters
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter
import ru.yandex.direct.core.entity.uac.model.UacFeedFilter
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterCondition
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterOperator
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.utils.CommonUtils
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.grut.objects.proto.client.Schema

data class Param(
    val masterCids: List<Long>?,
    val operatorUid: Long?
)

/**
 * Ваншот для сохранения фильтров на товарных кампаниях по фильтрам из заявок
 */
@Component
@Multilaunch
@Approvers("ali-al", "buhter", "kozobrodov")
class RestoreEcomUcFiltersOneshot @Autowired constructor(
    private val uacDbDefineService: UacDbDefineService,
    private val uacCampaignServiceHolder: UacCampaignServiceHolder,
    private val campaignService: CampaignService,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val shardHelper: ShardHelper,
    private val feedService: FeedService,
    private val performanceFilterService: PerformanceFilterService,
    private val dynamicTextAdTargetService: DynamicTextAdTargetService,
    private val performanceFilterStorage: PerformanceFilterStorage,
    private val adGroupService: AdGroupService,
    private val clientService: ClientService,
    private val grutApiService: GrutApiService,
    private val bannerService: BannerService,
    private val bannerModerateService: BannerModerateService
) : SimpleOneshot<Param, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(RestoreEcomUcFiltersOneshot::class.java)

        private const val DEFAULT_FILTER_NAME = "filter"
    }

    override fun validate(inputData: Param) = validateObject(inputData) {
        listProperty(inputData::masterCids)
            .check(CollectionConstraints.notEmptyCollection())
            .checkEach(NumberConstraints.greaterThan(0L))
    }

    override fun execute(inputData: Param, prevState: Void?): Void? {
        inputData.masterCids!!.forEach { handleSingleEcomCampaign(it, inputData.operatorUid) }
        return null
    }

    private fun handleSingleEcomCampaign(cid: Long, operatorUid: Long?) {
        val useGrut = uacDbDefineService.useGrutForDirectCampaignId(cid)
        val uacCampaignService = uacCampaignServiceHolder.getUacCampaignService(useGrut)

        val uacCampaign = uacCampaignService.getCampaignByDirectCampaignId(cid)

        if (uacCampaign == null || !CommonUtils.nvl(uacCampaign.isEcom, false)) {
            logger.error("Cannot find ecom campaign $cid")
            return
        }

        val shard = shardHelper.getShardByCampaignId(cid)
        val subcampaignIds = campaignService.getSubCampaignIdsWithMasterIds(setOf(cid)).keys
        val subcampaigns = campaignTypedRepository
            .getTypedCampaigns(shard, subcampaignIds)
            .filterIsInstance<CommonCampaign>()

        if (subcampaigns.size != 2) {
            logger.error("Unexpected number of subcampaigns for master cid $cid")
            return
        }
        val fixedUacCampaign = uacCampaign.fixFilterConditions()
        handleDynamicSubcampaign(fixedUacCampaign, subcampaigns.filterIsInstance<DynamicCampaign>().single(), operatorUid)
        handleSmartSubcampaign(fixedUacCampaign, subcampaigns.filterIsInstance<SmartCampaign>().single(), operatorUid)

        // Теперь надо пересохранить заявку с норм фильтром
        // Для простоты обновлять будет только и исключительно грутовые заявки
        if (useGrut) {
            grutApiService.briefGrutApi.updateBriefFull(
                Schema.TCampaign.newBuilder().apply {
                    meta = Schema.TCampaignMeta.newBuilder().setId(fixedUacCampaign.id.toIdLong()).build()
                    spec = UacGrutCampaignConverter.toCampaignSpec(fixedUacCampaign)
                }.build()
            )
        }
    }

    private fun sendBannersToModeration(clientId: ClientId, operatorUid: Long, adGroupIds: List<Long>) {
        val bannerIds = bannerService.getBannersByAdGroupIds(adGroupIds)
            .values
            .flatten()
            .map { it.id }
        if (bannerIds.isEmpty()) {
            logger.error("No banners found for groups $adGroupIds")
        }
        val moderationResult = bannerModerateService.moderateBanners(clientId, operatorUid, bannerIds)
        if (moderationResult.validationResult.hasAnyErrors()) {
            logger.error("Unable to send banners to moderation for $adGroupIds group," +
                " errors: ${moderationResult.validationResult.flattenErrors()[0]}")
        }
    }

    private fun handleDynamicSubcampaign(uacCampaign: UacYdbCampaign, dynamicCampaign: DynamicCampaign, operatorUid: Long?) {
        val adGroupIds = adGroupService.getAdGroupIdsByCampaignIds(setOf(dynamicCampaign.id))[dynamicCampaign.id]
        if (adGroupIds == null || adGroupIds.isEmpty()) {
            logger.error("Missing dynamic ad groups for dynamic campaign ${dynamicCampaign.id}")
            return
        }
        val operator = operatorUid ?: dynamicCampaign.agencyUid ?: dynamicCampaign.uid
        val clientId = ClientId.fromLong(dynamicCampaign.clientId)
        for (adGroupId in adGroupIds) {
            val criteria = DynamicTextAdTargetSelectionCriteria()
                .withAdGroupIds(adGroupId)
                .withStates(DynamicTextAdTargetState.ON)
            val filters = dynamicTextAdTargetService.getDynamicFeedAdTargets(clientId, operator, criteria, LimitOffset.maxLimited())
            if (filters == null || filters.isEmpty()) {
                val client = clientService.getClient(clientId)
                createDynamicFilter(client!!, operator, adGroupId, uacCampaign)
            }
        }
        sendBannersToModeration(clientId, operator, adGroupIds)
    }

    private fun handleSmartSubcampaign(uacCampaign: UacYdbCampaign, smartCampaign: SmartCampaign, operatorUid: Long?) {
        val adGroupIds = adGroupService.getAdGroupIdsByCampaignIds(setOf(smartCampaign.id))[smartCampaign.id]
        if (adGroupIds == null || adGroupIds.isEmpty()) {
            logger.error("Missing smart ad groups for smart campaign ${smartCampaign.id}")
            return
        }
        val operator = operatorUid ?: smartCampaign.agencyUid ?: smartCampaign.uid
        val clientId = ClientId.fromLong(smartCampaign.clientId)
        for (adGroupId in adGroupIds) {
            val filters = performanceFilterService.getPerformanceFilters(clientId, listOf(adGroupId))
                .values
                .flatten()
                .filter { !it.isDeleted && !it.isSuspended }
            if (filters.isEmpty()) {
                val client = clientService.getClient(clientId)
                createSmartFilter(client!!, operator, adGroupId, uacCampaign)
            }
        }
        sendBannersToModeration(clientId, operator, adGroupIds)
    }

    @Suppress("DuplicatedCode")
    private fun createSmartFilter(client: Client,
                                  operatorUid: Long,
                                  groupId: Long,
                                  ydbCampaign: UacYdbCampaign) {
        val clientId = ClientId.fromLong(client.id)
        val feed = feedService.getFeeds(clientId, setOf(ydbCampaign.feedId))!![0]
        val schema = performanceFilterStorage.getFilterSchema(feed.businessType, feed.feedType, feed.source)
        val filters = ydbCampaign.feedFilters?.toSmartFilters(groupId, DEFAULT_FILTER_NAME, feed, schema)
            ?: listOf(UacEcomConverter.defaultSmartFilter(groupId, ydbCampaign.name, feed))
        val result = performanceFilterService.add(clientId, operatorUid, filters, Applicability.FULL)
        if (result.validationResult.hasAnyErrors()) {
            logger.error("Cannot create smart filter: ${result.validationResult.flattenErrors()[0]}")
        }
    }

    @Suppress("DuplicatedCode")
    private fun createDynamicFilter(client: Client,
                                    operatorUid: Long,
                                    groupId: Long,
                                    ydbCampaign: UacYdbCampaign) {
        val clientId = ClientId.fromLong(client.id)
        val feed = feedService.getFeeds(clientId, setOf(ydbCampaign.feedId))!![0]
        val schema = performanceFilterStorage.getFilterSchema(feed.businessType, feed.feedType, feed.source)
        val filters = ydbCampaign.feedFilters?.toDynamicFilters(groupId, DEFAULT_FILTER_NAME, feed, schema)
            ?: listOf(UacEcomConverter.defaultDynamicFilter(groupId, ydbCampaign.name, feed))
        val result = dynamicTextAdTargetService.addDynamicFeedAdTargets(clientId, operatorUid, filters)
        if (result.validationResult.hasAnyErrors()) {
            logger.error("Cannot create dynamic filter: ${result.validationResult.flattenErrors()[0]}")
        }
    }

    private fun UacYdbCampaign.fixFilterConditions() = this.copy(
        feedFilters = this.feedFilters?.map { it.fixFilterConditions() }
    )

    private fun UacFeedFilter.fixFilterConditions() = this.copy(
        conditions = this.conditions.map { it.fix() }
    )

    private fun UacFeedFilterCondition.fix() =
        if (this.operator == UacFeedFilterOperator.RANGE && this.values != null && this.values!!.size == 2) {
            val newValue = "${this.values!![0]}-${this.values!![1]}"
            UacFeedFilterCondition(this.field, UacFeedFilterOperator.RANGE, newValue, listOf(newValue))
        } else {
            this
        }
}
