package ru.yandex.direct.oneshot.oneshots.performancefilter

import one.util.streamex.EntryStream
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFiltersQueryFilter
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatSerializer
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.DYNAMIC_CONDITIONS
import ru.yandex.direct.dbschema.ppc.Tables.PHRASES
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsDynamicStatusblgenerated
import ru.yandex.direct.dbschema.ppc.enums.AdgroupsPerformanceStatusblgenerated
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusbssynced
import ru.yandex.direct.dbschema.ppc.enums.BidsDynamicStatusbssynced
import ru.yandex.direct.dbschema.ppc.enums.BidsPerformanceStatusbssynced
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusbssynced
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class InputData(
    val adGroupIds: List<Long>,
)

/**
 * Ваншот для пересохранения фильтров смартов/ДО.
 * Нужен для обновления фильтров в базе и инициирования перегенерации при смене формата сериализации
 */
@Component
@Multilaunch
@Approvers("ali-al", "buhter", "kozobrodov")
class UpdatePerformanceFilterConditionsOneshot @Autowired constructor(
    private val shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
    private val performanceFilterRepository: PerformanceFilterRepository,
    private val dynamicTextAdTargetRepository: DynamicTextAdTargetRepository,
) : SimpleOneshot<InputData, Void> {
    private val serializer = PerformanceFilterConditionDBFormatSerializer.INSTANCE

    companion object {
        private val logger = LoggerFactory.getLogger(UpdatePerformanceFilterConditionsOneshot::class.java)
    }

    override fun validate(inputData: InputData): ValidationResult<InputData, Defect<Any>> {
        val vb = ItemValidationBuilder.of<InputData, Defect<Any>>(inputData)
        vb.item(inputData.adGroupIds, "adGroupIds")
                .check(CollectionConstraints.notEmptyCollection())
        return vb.result
    }

    override fun execute(inputData: InputData, prevState: Void?): Void? {
        logger.info("START")
        EntryStream.of(shardHelper.getClientIdsByAdGroupIds(inputData.adGroupIds))
                .mapValues(ClientId::fromLong)
                .forKeyValue { adGroupId, clientId ->
                    val shard = shardHelper.getShardByClientId(clientId)
                    logger.info("Processing adgroup {} on shard {}", adGroupId, shard)
                    val performanceFilters = performanceFilterRepository.getFilters(shard, PerformanceFiltersQueryFilter.newBuilder()
                            .withClientId(clientId)
                            .withAdGroupIds(listOf(adGroupId))
                            .withoutDeleted()
                            .build())
                    logger.info("Loaded {} performance filters: {}", performanceFilters.size, performanceFilters)
                    val conditionJsonByPerfFilterId = performanceFilters
                            .associateBy({ it.id }, { serializer.serialize(it.conditions) })
                    logger.info("Condition JSON by perf_filter_id: {}", conditionJsonByPerfFilterId)
                    val dynamicFeedAdTargets = dynamicTextAdTargetRepository.getDynamicFeedAdTargetsByAdGroupIds(shard,
                            clientId, listOf(adGroupId), false, LimitOffset.maxLimited())
                    logger.info("Loaded {} dynamic feed ad targets: {}", dynamicFeedAdTargets.size, dynamicFeedAdTargets)
                    val conditionJsonByDynCondId = dynamicFeedAdTargets
                            .associateBy({ it.dynamicConditionId }, { serializer.serialize(it.condition) })
                    logger.info("Condition JSON by dyn_cond_id: {}", conditionJsonByDynCondId)
                    dslContextProvider.ppcTransaction(shard) { conf ->
                        conf.dsl().update(BIDS_PERFORMANCE)
                                .set(BIDS_PERFORMANCE.CONDITION_JSON, makeCaseStatement(BIDS_PERFORMANCE.PERF_FILTER_ID,
                                        BIDS_PERFORMANCE.CONDITION_JSON, conditionJsonByPerfFilterId))
                                .set(BIDS_PERFORMANCE.STATUS_BS_SYNCED, BidsPerformanceStatusbssynced.No)
                                .where(BIDS_PERFORMANCE.PERF_FILTER_ID.`in`(conditionJsonByPerfFilterId.keys))
                                .execute()
                        conf.dsl().update(DYNAMIC_CONDITIONS.join(BIDS_DYNAMIC)
                                .on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID)))
                                .set(DYNAMIC_CONDITIONS.CONDITION_JSON, makeCaseStatement(DYNAMIC_CONDITIONS.DYN_COND_ID,
                                        DYNAMIC_CONDITIONS.CONDITION_JSON, conditionJsonByDynCondId))
                                .set(BIDS_DYNAMIC.STATUS_BS_SYNCED, BidsDynamicStatusbssynced.No)
                                .where(BIDS_DYNAMIC.DYN_COND_ID.`in`(conditionJsonByDynCondId.keys))
                                .execute()
                        conf.dsl().update(ADGROUPS_PERFORMANCE)
                                .set(ADGROUPS_PERFORMANCE.STATUS_BL_GENERATED, AdgroupsPerformanceStatusblgenerated.No)
                                .where(ADGROUPS_PERFORMANCE.PID.eq(adGroupId))
                                .execute()
                        conf.dsl().update(ADGROUPS_DYNAMIC)
                                .set(ADGROUPS_DYNAMIC.STATUS_BL_GENERATED, AdgroupsDynamicStatusblgenerated.No)
                                .where(ADGROUPS_DYNAMIC.PID.eq(adGroupId))
                                .execute()
                        conf.dsl().update(PHRASES)
                                .set(PHRASES.STATUS_BS_SYNCED, PhrasesStatusbssynced.No)
                                .where(PHRASES.PID.eq(adGroupId))
                                .execute()
                        conf.dsl().update(BANNERS)
                                .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                                .where(BANNERS.PID.eq(adGroupId))
                                .execute()
                    }
                }
        logger.info("FINISH")
        return null
    }
}
