package ru.yandex.direct.oneshot.oneshots.phones;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.jooq.CaseWhenStep;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneMapping;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_PHONES;

@Component
@Multilaunch
@Approvers("maxlog")
public class PhoneNumberRecordUpdaterOneshot implements ShardedOneshot<Void, Void> {

    private static final int CHUNK_SIZE = 1000;
    private final Logger logger = LoggerFactory.getLogger(PhoneNumberRecordUpdaterOneshot.class);

    private final DslContextProvider dslContextProvider;
    private final ClientPhoneRepository clientPhoneRepository;
    private final BannerCommonRepository bannerCommonRepository;

    public PhoneNumberRecordUpdaterOneshot(
            DslContextProvider dslContextProvider,
            ClientPhoneRepository clientPhoneRepository,
            BannerCommonRepository bannerCommonRepository
    ) {
        this.dslContextProvider = dslContextProvider;
        this.clientPhoneRepository = clientPhoneRepository;
        this.bannerCommonRepository = bannerCommonRepository;
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }

    @Nullable
    @Override
    public Void execute(Void inputData, Void prevState, int shard) {
        var manualPhoneNumbersById = getManualPhoneNumbersById(shard);
        logger.info("Start update {} phones", manualPhoneNumbersById.size());

        Map<Long, String> formattedManualPhoneNumbersById = new HashMap<>();
        manualPhoneNumbersById.forEach((id, oldNumber) -> {
            PhoneNumber oldPhoneNumber = ClientPhoneMapping.phoneNumberFromDb(oldNumber);
            String newNumber = ClientPhoneMapping.phoneNumberToDb(oldPhoneNumber);
            if (!oldNumber.equals(newNumber)) {
                logger.info("Phone changes: {} -> {}", oldNumber, newNumber);
                formattedManualPhoneNumbersById.put(id, newNumber);
            }
        });

        if (formattedManualPhoneNumbersById.isEmpty()) {
            logger.info("No phones for updating");
            return null;
        }

        updatePhones(shard, formattedManualPhoneNumbersById);
        resetStatusBsSynced(shard, formattedManualPhoneNumbersById.keySet());
        logger.info("Finish update phones");
        return null;
    }

    @QueryWithoutIndex("На каждом шарде до 1500 записей")
    Map<Long, String> getManualPhoneNumbersById(int shard) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_PHONES.CLIENT_PHONE_ID, CLIENT_PHONES.PHONE)
                .from(CLIENT_PHONES)
                .where(CLIENT_PHONES.PHONE_TYPE.eq(ClientPhoneType.toSource(ClientPhoneType.MANUAL)))
                .fetchMap(CLIENT_PHONES.CLIENT_PHONE_ID, CLIENT_PHONES.PHONE);
    }

    private void updatePhones(int shard, Map<Long, String> phoneById) {
        CaseWhenStep<Long, String> values = DSL.choose(CLIENT_PHONES.CLIENT_PHONE_ID).mapValues(phoneById);
        dslContextProvider.ppc(shard)
                .update(CLIENT_PHONES)
                .set(CLIENT_PHONES.PHONE, values)
                .where(CLIENT_PHONES.CLIENT_PHONE_ID.in(phoneById.keySet()))
                .execute();
    }

    private void resetStatusBsSynced(int shard, Set<Long> phoneIds) {
        logger.info("Start reset statusBsSynced");
        var bannerIdsByPhoneId = clientPhoneRepository.getBannerIdsByPhoneId(shard, phoneIds);
        Set<Long> bannerIds = StreamEx.ofValues(bannerIdsByPhoneId).flatMap(StreamEx::of).toSet();
        for (var chunkedBannerIds : Iterables.partition(bannerIds, CHUNK_SIZE)) {
            bannerCommonRepository.resetStatusBsSyncedByIds(shard, chunkedBannerIds);
            logger.info("Reset statusBsSynced for {} banners", chunkedBannerIds.size());
        }
        logger.info("End reset statusBsSynced");
    }
}
