package ru.yandex.direct.oneshot.oneshots.promocodes;

import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.promocodes.repository.PromocodeDomainsRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.Retries;
import ru.yandex.direct.oneshot.worker.def.SafeOneshot;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.StringDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbschema.ppcdict.tables.PromocodeDomains.PROMOCODE_DOMAINS;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Component
@SafeOneshot
@Multilaunch
@Retries(2)
public class DeleteUnsupportedPromocodeDomainsOneShot
        implements SimpleOneshot<DeleteUnsupportedPromocodeDomainsInputData, Void> {
    private static final Logger logger = LoggerFactory.getLogger(DeleteUnsupportedPromocodeDomainsOneShot.class);
    private final PromocodeDomainsRepository repository;
    private static final String ALPHA_NUM_REGEX = "^[a-zA-Z0-9]*$";
    private final DslContextProvider dslContextProvider;

    @Autowired DeleteUnsupportedPromocodeDomainsOneShot(PromocodeDomainsRepository repository,
                                                        DslContextProvider dslContextProvider) {
        this.repository = repository;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    @SuppressWarnings("rawtypes")
    public ValidationResult<DeleteUnsupportedPromocodeDomainsInputData, Defect> validate(
            DeleteUnsupportedPromocodeDomainsInputData inputData
    ) {
        ItemValidationBuilder<DeleteUnsupportedPromocodeDomainsInputData, Defect> ivb =
                ItemValidationBuilder.of(inputData);
        ivb.item(inputData.getPromocodes(), "promocodes")
                .check(notNull())
                .check(eachNotNull(), When.isValid())
                .check(Constraint.fromPredicate(
                        list -> list.stream().allMatch(str -> str.matches(ALPHA_NUM_REGEX)),
                        StringDefects.mustContainLettersOrDigitsOrPunctuations() // про пунктуацию здесь обманываем
                ), When.isValid());
        return ivb.getResult();
    }

    @Nullable
    @Override
    public Void execute(DeleteUnsupportedPromocodeDomainsInputData inputData, @Nullable Void state) {
        var promocodes = inputData.getPromocodes();
        var dataToDelete = repository.getPromocodeDomains(promocodes);
        logger.info("Will delete the following promocode_domains data: {}", dataToDelete);
        dslContextProvider.ppcdict() // у репозитория нет метода удаления по id
                .deleteFrom(PROMOCODE_DOMAINS)
                .where(PROMOCODE_DOMAINS.PROMOCODE.in(dataToDelete.keySet()))
                .execute();
        return null;
    }
}
