package ru.yandex.direct.oneshot.oneshots.remoderatecpmbanners

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.model.CpmBanner
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SafeOneshot
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect

data class InputData(
    val bannersIds: List<Long>
)

/**
 * для списка баннеров будет сбрасывать statusModerate для banners в Ready
 * Для баннеров с типом banner_type=cpm_banner
 */
@Component
@Multilaunch
@SafeOneshot
open class RemoderateCpmBannersOneshot (
    private val shardHelper: ShardHelper,
    private val bannerModerationRepository: BannerModerationRepository,
    private val bannerTypedRepository: BannerTypedRepository,
) : SimpleOneshot<InputData, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(RemoderateCpmBannersOneshot::class.java)
        private const val LIMIT = 100
    }

    override fun validate(inputData: InputData) =
        ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.bannersIds, "bannersIds")
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.minListSize(1))
                .check(CollectionConstraints.maxListSize(LIMIT))
        }.result

    override fun execute(inputData: InputData, prevState: Void?): Void? {
        val groupByShard = shardHelper.groupByShard(inputData.bannersIds, ShardKey.BID) { it }
        for (shardedIdsEntry in groupByShard.stream()) {
            val shard = shardedIdsEntry.key
            val bids = shardedIdsEntry.value
            var banners = bannerTypedRepository.getSafely(shard, bids, CpmBanner::class.java)
            var bid2remoderate = banners.map { it.id }
            logger.info("Reset moderation status for banners: $bid2remoderate")
            bannerModerationRepository.requestRemoderation(shard, bid2remoderate)
        }
        return null
    }
}
