package ru.yandex.direct.oneshot.oneshots.remoderatesmartcreatives

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect

data class InputData(
    val creativeIds: List<Long>
)

/**
 * Выполняет переотправку в Модерацию указанные смартовые креативы
 * (не более 2000 штук за раз)
 */
@Component
@Multilaunch
@Approvers("elwood", "ulyashevda")
open class RemoderateSmartCreativesOneshot(
    private val shardHelper: ShardHelper,
    private val creativeRepository: CreativeRepository
) : SimpleOneshot<InputData, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(RemoderateSmartCreativesOneshot::class.java)

        /**
         * Мы не можем переотправлять слишком много креативов, чтобы не заддосить Модерацию
         * Текущий лимит согласован с serdyuk-s@
         */
        private const val LIMIT = 2_000
        private const val CHUNK_SIZE = 100
    }

    override fun validate(inputData: InputData) =
        ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.creativeIds, "creativeIds")
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.minListSize(1))
                .check(CollectionConstraints.maxListSize(LIMIT))
        }.result

    override fun execute(inputData: InputData, prevState: Void?): Void? {
        val groupByShard = shardHelper.groupByShard(inputData.creativeIds, ShardKey.CREATIVE_ID, { it })
        for (shardedIdsEntry in groupByShard.chunkedBy(CHUNK_SIZE).stream()) {
            val shard = shardedIdsEntry.key
            val creativeIds = shardedIdsEntry.value

            // Сначала сбросим статус в Error, чтобы далее можно было его сменить на Ready
            logger.info("Reset moderation status to Error for creatives: $creativeIds")
            creativeRepository.setStatusModerate(shard, creativeIds, PerfCreativesStatusmoderate.Error)

            // И меняем на Ready
            logger.info("Send creatives to moderation: $creativeIds")
            creativeRepository.sendCreativesToModeration(shard, creativeIds)
        }
        return null
    }
}
