package ru.yandex.direct.oneshot.oneshots.remove_duplicate_banners_after_migration_to_grut

import com.google.common.collect.Lists
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.DELETE_DUPLICATE_BANNERS_AFTER_MIGRATION_TO_GRUT_RELAX_TIME
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.oneshots.remove_duplicate_banners_after_migration_to_grut.repository.OneshotRemoveDuplicateBannersRepository
import ru.yandex.direct.oneshot.oneshots.uc.UacConverterYtRepository
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.MigrationResult
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable

data class InputData(
    val ytCluster: YtCluster,
    val tablePath: String,
    val operator: Long,
)

data class State(
    var lastRow: Long,
    var deletedBannersCount: Int,
)

/**
 * Ваншот для удаления дублей баннеров, добавленных после миграции DIRECT-152917 в grut
 *
 * В yt таблице должна быть следующая колонка: ydb_cid
 */
@Component
@Multilaunch
@Approvers("mspirit", "dimitrovsd", "khuzinazat", "a-dubov", "ppalex")
@Retries(5)
@PausedStatusOnFail
class RemoveDuplicateBannersOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val oneshotRemoveDuplicateBannersRepository: OneshotRemoveDuplicateBannersRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val shardHelper: ShardHelper,
    private val uacBannerService: UacBannerService,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val adGroupRepository: AdGroupRepository,
) : SimpleOneshot<InputData, State?> {
    companion object {
        private val logger = LoggerFactory.getLogger(RemoveDuplicateBannersOneshot::class.java)
        private const val DELETE_CHUNK_SIZE = 1_000
        private const val DEFAULT_RELAX_TIME_BETWEEN_DELETE_ITERATIONS_IN_SEC = 5L
        private val resultTablePath = "//home/direct/test/pavelkataykin/RemoveDuplicateBannersOneshotResult_${ru.yandex.direct.env.Environment.getCached().name.lowercase()}"
    }

    override fun validate(inputData: InputData) = validateObject(inputData) {
        property(inputData::ytCluster) {
            check(CommonConstraints.notNull())
        }
        property(inputData::tablePath) {
            check(CommonConstraints.notNull())
            check(
                Constraint.fromPredicate(
                    {
                        ytProvider.getOperator(inputData.ytCluster).exists(YtTable(it))
                    },
                    CommonDefects.objectNotFound()
                ), When.isValid()
            )
        }
    }

    override fun execute(
        inputData: InputData,
        prevState: State?
    ): State? {
        val state = prevState ?: State(0L, 0)

        val startRow = state.lastRow
        val lastRow = startRow + 1
        logger.info("Iteration starts with rows [$startRow, $lastRow)")
        uacConverterYtRepository.createResultTableIfNotExists(resultTablePath)

        val relaxTimeProperty = ppcPropertiesSupport.get(DELETE_DUPLICATE_BANNERS_AFTER_MIGRATION_TO_GRUT_RELAX_TIME)
        val relaxTimeBetweenDeleteIterations = relaxTimeProperty.getOrDefault(DEFAULT_RELAX_TIME_BETWEEN_DELETE_ITERATIONS_IN_SEC)

        val adGroupIdsChunk = oneshotRemoveDuplicateBannersRepository
            .getAdGroupIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)

        if (adGroupIdsChunk.isEmpty()) {
            logger.info("Delete process completed! Total deletions ${state.deletedBannersCount}")
            return null
        }
        if (adGroupIdsChunk.size > 1) {
            throw IllegalStateException("More than one adgroups $adGroupIdsChunk")
        }
        val adGroupId = adGroupIdsChunk[0]


        val shard = shardHelper.getShardByGroupId(adGroupId)
        val campaignId = adGroupRepository.getCampaignIdsByAdGroupIds(shard, listOf(adGroupId))[adGroupId]!!
        val clientId = ClientId.fromLong(shardHelper.getClientIdByCampaignId(campaignId))
        val duplicateBannerIds = oneshotRemoveDuplicateBannersRepository
            .getDuplicateBannerIdsByAdGroupId(resultTablePath, adGroupId)

        try {
            for (bannerIdsChunkToDelete in Lists.partition(duplicateBannerIds, DELETE_CHUNK_SIZE)) {


                var deletedBannersCount = 0
                // Останавливаем и удаляем баннера
                deletedBannersCount += archiveOrDeleteBanners(inputData.operator, clientId, bannerIdsChunkToDelete)
                state.deletedBannersCount += deletedBannersCount

                logger.info("Deleted $deletedBannersCount banners from $bannerIdsChunkToDelete. " +
                    "Iteration finished, sleep for $relaxTimeBetweenDeleteIterations seconds")
                Thread.sleep(relaxTimeBetweenDeleteIterations * 1000)
            }
            if (duplicateBannerIds.isNotEmpty()) {
                uacConverterYtRepository
                    .writeResults(resultTablePath, listOf(MigrationResult(adGroupId, true, "")))
            }

        } catch (e: Exception) {
            uacConverterYtRepository
                .writeResults(resultTablePath, listOf(MigrationResult(adGroupId, false, e.message)))
        }

        return State(lastRow, state.deletedBannersCount)
    }

    /**
     * Архивирование или удаление баннеров
     * Взято из BannerCleanJobService
     */
    private fun archiveOrDeleteBanners(
        operatorUid: Long,
        clientId: ClientId,
        bannerIdsToRemove: List<Long>,
    ): Int {
        // Останавливаем баннеры
        val suspendResult = uacBannerService.suspendResumeBanners(clientId, operatorUid, false, bannerIdsToRemove)
        suspendResult.logErrors("banners suspension")

        val (deleteMassResult, archiveMassResult) = uacBannerService
            .deleteAndArchiveBanners(operatorUid, clientId, bannerIdsToRemove.toSet())
        deleteMassResult.logErrors("removing banners")
        archiveMassResult.logErrors("archiving banners")

        val successfulResultBannerIds = mutableSetOf<Long>()
        successfulResultBannerIds.addAll(getSuccessResults(deleteMassResult) + getSuccessResults(archiveMassResult))

        val failedResultBannerIds = bannerIdsToRemove - successfulResultBannerIds

        // Обновляем статусы баннеров в ydb
        if (successfulResultBannerIds.isNotEmpty()) {
            uacYdbDirectAdRepository.updateStatusByDirectAdIds(successfulResultBannerIds, DirectAdStatus.DELETED)
        }
        if (failedResultBannerIds.isNotEmpty()) {
            uacYdbDirectAdRepository.updateStatusByDirectAdIds(failedResultBannerIds, DirectAdStatus.DELETED)
        }
        return successfulResultBannerIds.size
    }

    private fun getSuccessResults(massResult: MassResult<Long>?): Set<Long> {
        return if (massResult?.result == null) {
            emptySet()
        } else massResult.result
            .filterNotNull()
            .filter { it.isSuccessful }
            .mapNotNull { it.result }
            .toSet()
    }

    private fun MassResult<Long>.logErrors(action: String = "action") {
        if (!this.isSuccessful) {
            logger.error("Result of $action was unsuccessful: {0}", this.validationResult.flattenErrors())
        }
    }
}
