package ru.yandex.direct.oneshot.oneshots.remove_duplicate_banners_after_migration_to_grut.repository

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.oneshots.uc.UacConverterYtRepository
import ru.yandex.direct.oneshot.oneshots.uc.uacconverter.MigrationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

class InputTableRow : YtTableRow(listOf(PID)) {
    companion object {
        private val PID = YtField("pid", Long::class.java)
    }

    val pid: Long
        get() = valueOf(PID)
}

@Component
class OneshotRemoveDuplicateBannersRepository(
    private val ytProvider: YtProvider,
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val uacYdbDirectAdGroupRepository: UacYdbDirectAdGroupRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val adGroupRepository: AdGroupRepository,
    private val shardHelper: ShardHelper,
    private val grutApiService: GrutApiService,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(OneshotRemoveDuplicateBannersRepository::class.java)
    }

    /**
     * Получение записей из Yt таблички, по строкам
     *
     * @param ytCluster кластер
     * @param tablePath таблица
     * @param startRow  номер строки, с которой начинам считывать (включительно)
     * @param lastRow   номер строки, на которой заканчиваем считывать (исключая)
     */
    fun getAdGroupIdsFromYtTable(ytCluster: YtCluster,
                                 tablePath: String,
                                 startRow: Long,
                                 lastRow: Long
    ): List<Long> {
        val adGroupIds = mutableListOf<Long>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(
                YtTable(tablePath),
                { adGroupIds.add(it.pid) },
                InputTableRow(),
                startRow,
                lastRow)
        return adGroupIds
    }

    fun getDuplicateBannerIdsByAdGroupId(resultTablePath: String, adGroupId: Long): List<Long> {

        val uacAdGroup = uacYdbDirectAdGroupRepository.getDirectAdGroupByDirectAdGroupId(adGroupId)
        if (uacAdGroup == null) {
            val message = "AdGroup $adGroupId not fount in ydb"
            logger.warn(message)
            val migrationResult = MigrationResult(adGroupId, false, message)
            uacConverterYtRepository.writeResults(resultTablePath, listOf(migrationResult))
            return listOf()
        }
        logger.info("Start handle adGroup $adGroupId ydb id ${uacAdGroup.id}")
        val shard = shardHelper.getShardByGroupId(uacAdGroup.directAdGroupId)
        val directCampaignId = adGroupRepository.getCampaignIdsByAdGroupIds(shard, listOf(uacAdGroup.directAdGroupId))[uacAdGroup.directAdGroupId]
        logger.info("Direct campaign id $directCampaignId for adGroup id ${uacAdGroup.directAdGroupId}")
        val grutCampaign = getGrutCampaign(directCampaignId!!)
        if (grutCampaign == null) {
            val message = "Campaign $directCampaignId for adGroup id $adGroupId not found in grut, not migrate"
            logger.warn(message)
            val migrationResult = MigrationResult(adGroupId, false, message)
            uacConverterYtRepository.writeResults(resultTablePath, listOf(migrationResult))
            return listOf()
        }
        logger.info("Campaign $directCampaignId for adGroup id $adGroupId found in grut, not migrate")
        val banners = getYdbBanners(uacAdGroup.id)
        val directBids = banners.mapNotNull { banner -> banner.directAdId }
        val grutBannerIds = grutApiService.briefBannerGrutApi.getExistingObjects(directBids).toSet()

        val bannersToDelete = directBids.filterNot { bid -> grutBannerIds.contains(bid) }
        logger.info("Banners for campaign $directCampaignId for adGroup $adGroupId to delete $bannersToDelete")
        if (bannersToDelete.isEmpty()) {
            val message = "No banners to delete for campaign $directCampaignId, adgroup $adGroupId"
            logger.info(message)
            val migrationResult = MigrationResult(adGroupId, true, message)
            uacConverterYtRepository.writeResults(resultTablePath, listOf(migrationResult))
        }
        return bannersToDelete
    }

    private fun getYdbBanners(ydbAdGroupId: String): List<UacYdbDirectAd> {
        val allYdbDirectAds = mutableListOf<UacYdbDirectAd>()

        var fromUacAdId = 0L
        while (true) {
            val ydbDirectAdsFromYdb = uacYdbDirectAdRepository
                .getByDirectAdGroupId(listOf(ydbAdGroupId), fromUacAdId, 990L)

            allYdbDirectAds.addAll(ydbDirectAdsFromYdb)
            if (ydbDirectAdsFromYdb.size < 990L) {
                break
            }

            // Здесь нельзя сравнивать и брать наибольший, т.к. значения id выше long.max - отрицательные
            fromUacAdId = ydbDirectAdsFromYdb.last().id.toIdLong()
        }
        return allYdbDirectAds
    }

    private fun getGrutCampaign(campaignId: Long) =
        grutApiService.briefGrutApi.getBrief(campaignId)

}





