package ru.yandex.direct.oneshot.oneshots.removeincorrectbids;

import java.util.List;

import com.google.common.collect.Iterables;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.base.ShardedOneshotWithoutInput;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_ARC;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_BASE;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@Approvers({"ppalex"})
@Multilaunch
public class RemoveIncorrectBids extends ShardedOneshotWithoutInput {
    private static final Logger logger = LoggerFactory.getLogger(RemoveIncorrectBids.class);
    public static final String YQL_QUERY =
            "select cid, id from `//home/direct/mysql-sync/current/ppc:%d/straight/bids_arc` where pid = 0";
    public static final int CHUNK_SIZE = 500;

    private final DslContextProvider dslContextProvider;
    private final YtProvider ytProvider;

    public RemoveIncorrectBids(DslContextProvider dslContextProvider, YtProvider ytProvider) {
        this.dslContextProvider = dslContextProvider;
        this.ytProvider = ytProvider;
    }

    @Override
    public void execute(int shard) {
        var toDeleteFromBids = dslContextProvider.ppc(shard)
                .select(BIDS.ID)
                .from(BIDS)
                .where(BIDS.PID.eq(0L))
                .fetch(BIDS.ID);

        for (var chunk : Iterables.partition(toDeleteFromBids, CHUNK_SIZE)) {
            dslContextProvider.ppc(shard).delete(BIDS).where(BIDS.ID.in(chunk)).execute();
            logger.info("Deleted {} records from bids with bids.id = {}", chunk.size(), chunk);
        }

        var toDeleteFromBidsBase = dslContextProvider.ppc(shard)
                .select(BIDS_BASE.BID_ID)
                .from(BIDS_BASE)
                .where(BIDS_BASE.PID.eq(0L))
                .fetch(BIDS_BASE.BID_ID);

        for (var chunk : Iterables.partition(toDeleteFromBidsBase, CHUNK_SIZE)) {
            dslContextProvider.ppc(shard).delete(BIDS_BASE).where(BIDS_BASE.BID_ID.in(chunk)).execute();
            logger.info("Deleted {} records from bids_base with bids_base.bid_id = {}", chunk.size(), chunk);
        }

        var toDeleteFromBidsArc = mapList(
                getCampIdsAndBidIdsFromBidsArc(String.format(YQL_QUERY, shard)),
                p -> row(p.getLeft(), 0L, p.getRight())
        );

        for (var chunk : Iterables.partition(toDeleteFromBidsArc, CHUNK_SIZE)) {
            dslContextProvider.ppc(shard)
                    .delete(BIDS_ARC)
                    .where(row(BIDS_ARC.CID, BIDS_ARC.PID, BIDS_ARC.ID).in(chunk))
                    .execute();
            logger.info("Deleted {} records from bids_arc with (bids_arc.cid, bids_arc.pid, bids_arc.id) = {}",
                    chunk.size(), chunk);
        }
    }

    private List<Pair<Long, Long>> getCampIdsAndBidIdsFromBidsArc(String query) {
        return ytProvider.getOperator(YtCluster.HAHN, YtSQLSyntaxVersion.SQLv1)
                .yqlQuery(query, row -> Pair.of(row.getLong("cid"), row.getLong("id")));
    }
}
