package ru.yandex.direct.oneshot.oneshots.resend_image_banners_in_old_scheme

import java.time.Duration
import kotlin.math.max
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.oneshot.oneshots.resend_image_banners_in_old_scheme.repository.ResendImageBannersInOldSchemeRepository
import ru.yandex.direct.oneshot.oneshots.uc.UacConverterYtRepository
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster

data class Param(
    val ytCluster: YtCluster,
    val tablePath: String,
    val operator: Long,
)

data class State(
    val lastRow: Long,
    val countOfResendBanners: Long,
)

data class BannerImageData(
    val bannerId: Long,
    val imageId: Long,
)


/**
 * Ваншот для отправки картиночных баннеров в старой схеме (в виде двух - картиночной и бескартиночной версиях)
 * Отбирает из переданного списка только те баннеры у которых есть флаг banner_images.opt = single_ad_to_bs
 * Чтобы баннеры отправились в старой схеме нужно чтобы была выключена фича SINGLE_IMAGE_AD_TO_BS
 *
 * Пример параметра запуска: {"tablePath": "//home/direct/test/pavelkataykin/rmp_fix/total", "ytCluster": "hahn"}
 * В Yt табличке ваншот будет собирать список id кампаний по колонке 'cid'
 */
@Component
@Approvers("ppalex", "mspirit", "khuzinazat", "a-dubov", "maxlog", "gerdler", "bratgrim", "andreypav")
@Multilaunch
@PausedStatusOnFail
class ResendImageBannersInOldSchemeOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val uacConverterYtRepository: UacConverterYtRepository,
    private val shardHelper: ShardHelper,
    private val resendImageBannersInOldSchemeRepository: ResendImageBannersInOldSchemeRepository,
    private val bsResyncService: BsResyncService,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : SimpleOneshot<Param, State?> {
    companion object {
        private const val CHUNK_CAMPAIGNS_SIZE = 2000
        private const val CHUNK_IMAGE_BANNERS_SIZE = 2000
        private const val DEFAULT_SLEEP_TIME = 5000L
        private const val MIN_SLEEP_TIME = 50L
        private const val MAX_COUNT_OF_ITERATIONS = 500
        private val logger = LoggerFactory.getLogger(this::class.java)!!
    }

    private val sleepTimeProperty = ppcPropertiesSupport
        .get(PpcPropertyNames.SINGLE_IMAGE_AD_TO_BS_IN_OLD_SCHEME_SLEEP, Duration.ofSeconds(60))

    override fun validate(
        inputData: Param
    ): ValidationResult<Param, Defect<*>> {
        val validateBuilder = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(
            ytProvider,
            validateBuilder,
            inputData.ytCluster,
            inputData.tablePath
        )
    }

    override fun execute(
        inputData: Param,
        prevState: State?
    ): State? {
        var countOfResendBanners = prevState?.countOfResendBanners ?: 0
        val startRow = prevState?.lastRow ?: 0
        val lastRow = startRow + CHUNK_CAMPAIGNS_SIZE
        val chunkOfCampaignIds = uacConverterYtRepository
            .getCampaignIdsFromYtTable(inputData.ytCluster, inputData.tablePath, startRow, lastRow)
        if (chunkOfCampaignIds.isEmpty()) {
            logger.info("All campaigns processed, $countOfResendBanners banners resent")
            return null
        }

        val sleepTime = max(sleepTimeProperty.getOrDefault(DEFAULT_SLEEP_TIME), MIN_SLEEP_TIME)

        val shardToCampainIds = shardHelper.groupByShard(chunkOfCampaignIds, ShardKey.CID).shardedDataMap
        shardToCampainIds.forEach { (shard, campaignIds) ->
            campaignIds.forEach { campaignId ->
                countOfResendBanners += resendBanners(shard, campaignId, sleepTime)
            }
        }

        return if (chunkOfCampaignIds.size < CHUNK_CAMPAIGNS_SIZE) {
            logger.info("All campaigns processed, $countOfResendBanners banners resent")
            null
        } else {
            State(lastRow, countOfResendBanners)
        }
    }

    /**
     * Сбрасываем у картиночных баннеров переданной кампании флаг single_ad_to_bs и переотправляем в БК
     */
    private fun resendBanners(
        shard: Int,
        campaignId: Long,
        sleepTime: Long,
    ): Int {
        var fromImageId = 0L
        var iterationNumber = 0
        var countOfResendBanners = 0
        while (true) {
            // Макс. количество обработанных баннеров = 1_000_000
            if (++iterationNumber > MAX_COUNT_OF_ITERATIONS) {
                throw RuntimeException("shard=$shard, cid=$campaignId, the iteration limit of $MAX_COUNT_OF_ITERATIONS reached")
            }

            // Собираем id баннеров, которые ранее были отправлены в новой схеме (в родительском баннере)
            val groupIdToBannerIdData = resendImageBannersInOldSchemeRepository
                .getImageBannerIdsWithSingleAdFlag(shard, campaignId, fromImageId, CHUNK_IMAGE_BANNERS_SIZE)
            val imageIds = groupIdToBannerIdData.values
                .flatten()
                .map { it.imageId }
                .toSet()
            if (imageIds.isEmpty()) {
                logger.info("shard=$shard, cid=$campaignId, campaigns processed, $countOfResendBanners banners resent")
                break
            }

            // Сбрасываем флаг banner_images.opts.single_ad_to_bs и БК'шный bannerId у картиночных баннеров
            resendImageBannersInOldSchemeRepository.removeBannerIdAndSingleAdFlagFromImageBanner(shard, imageIds)

            // Кладем в очередь на переотправку в БК
            val bsResyncItems = groupIdToBannerIdData
                .map { (groupId, bannerImageData) ->
                    bannerImageData.map {
                        fromImageId = fromImageId.coerceAtLeast(it.imageId)
                        BsResyncItem(BsResyncPriority.UPDATE_BANNER_WITH_ADDITIONS, campaignId, it.bannerId, groupId)
                    }
                }
                .flatten()
            bsResyncService.addObjectsToResync(bsResyncItems)

            logger.info("shard=$shard, cid=$campaignId, iteration finished, sleep for $sleepTime milliseconds")
            Thread.sleep(sleepTime)

            countOfResendBanners += imageIds.size
            if (imageIds.size < CHUNK_IMAGE_BANNERS_SIZE) {
                logger.info("shard=$shard, cid=$campaignId, campaigns processed, $countOfResendBanners banners resent")
                break
            }
        }
        return countOfResendBanners
    }
}

