package ru.yandex.direct.oneshot.oneshots.reset_acceptance_of_offer_for_turkish_clients;

import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;

import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.ClientsWorkCurrency;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsIsOfferAccepted;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS_OPTIONS;

/**
 * Сбрасывает у всех пользователей из Турции (или с валютой TRY) флаг принятия оферты
 */
@Component
@Multilaunch
@Approvers({"mexicano", "ppalex", "ajkon"})
public class ResetAcceptanceOfOfferForTurkishClientsOneshot implements ShardedOneshot<Void, Void> {

    private static final Logger logger = LoggerFactory.getLogger(ResetAcceptanceOfOfferForTurkishClientsOneshot.class);
    private static final int CHUNK_SIZE = 10_000;

    private final DslContextProvider dslContextProvider;

    public ResetAcceptanceOfOfferForTurkishClientsOneshot(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }

    @Nullable
    @Override
    public Void execute(Void inputData, Void prevState, int shard) {

        List<Long> userIdsToChange = getUserIdFromTurkeyAndWithOfferAccepted(shard);
        if (userIdsToChange.isEmpty()) {
            logger.info("shard: {}, There are no users to reset the acceptance of the offer", shard);
            return null;
        }
        logger.info("shard: {}, Start reset the acceptance of the offer for {} users", shard, userIdsToChange.size());

        int updatedCount = 0;
        for (List<Long> chunkOfUserIds : Lists.partition(userIdsToChange, CHUNK_SIZE)) {
            updatedCount += resetOfferAcceptanceByUserIds(shard, chunkOfUserIds);
        }

        logger.info("shard: {}, Reset offer acceptance for {} users", shard, updatedCount);
        return null;
    }

    /**
     * Собирает все uid пользователей с валютой TRY или страной Турция (country_region_id = 983) и с принятой офертой
     */
    private List<Long> getUserIdFromTurkeyAndWithOfferAccepted(int shard) {
        return dslContextProvider.ppc(shard)
                .select(USERS.UID)
                .from(USERS)
                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(USERS.CLIENT_ID))
                .join(USERS_OPTIONS).on(USERS_OPTIONS.UID.eq(USERS.UID))
                .where((CLIENTS.COUNTRY_REGION_ID.eq(Region.TURKEY_REGION_ID))
                        .or(CLIENTS.WORK_CURRENCY.eq(ClientsWorkCurrency.TRY)))
                .and(USERS_OPTIONS.IS_OFFER_ACCEPTED.eq(UsersOptionsIsOfferAccepted.Yes))
                .fetch(USERS.UID);
    }

    /**
     * Сбрасывает флаг принятия оферты у пользователей c id {@param userIds}
     */
    private int resetOfferAcceptanceByUserIds(int shard, Collection<Long> userIds) {
        logger.info("shard: {}, Reset offer acceptance for user ids: {}", shard, userIds);
        return dslContextProvider.ppc(shard)
                .update(USERS_OPTIONS)
                .set(USERS_OPTIONS.IS_OFFER_ACCEPTED, UsersOptionsIsOfferAccepted.No)
                .where(USERS_OPTIONS.UID.in(userIds))
                .and(USERS_OPTIONS.IS_OFFER_ACCEPTED.eq(UsersOptionsIsOfferAccepted.Yes))
                .execute();
    }
}
