package ru.yandex.direct.oneshot.oneshots.reset_no_title_substitute;

import java.util.List;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Ваншот вычищает из campaign.opts флаг no_title_substitute
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers({"ssdmitriev", "ruslansd", "buhter"})
public class ResetNoTitleSubstituteFlagFromCampaignsOneshot implements ShardedOneshot<Void,
        ResetNoTitleSubstituteFlagFromCampaignsState> {
    private static final Logger logger = LoggerFactory.getLogger(ResetNoTitleSubstituteFlagFromCampaignsOneshot.class);

    private static final int LIMIT = 10000;
    private static final List<CampaignType> TYPES_WITH_OPTION = List.of(
            CampaignType.CONTENT_PROMOTION,
            CampaignType.CPM_BANNER,
            CampaignType.CPM_PRICE,
            CampaignType.CPM_YNDX_FRONTPAGE,
            CampaignType.DYNAMIC,
            CampaignType.INTERNAL_AUTOBUDGET,
            CampaignType.INTERNAL_DISTRIB,
            CampaignType.INTERNAL_FREE,
            CampaignType.MCBANNER,
            CampaignType.MOBILE_CONTENT,
            CampaignType.TEXT);

    private final ResetNoTitleSubstituteFlagFromCampaignsRepository campaignTypedRepository;
    private final CampaignModifyRepository campaignModifyRepository;

    @Autowired
    public ResetNoTitleSubstituteFlagFromCampaignsOneshot(
            ResetNoTitleSubstituteFlagFromCampaignsRepository campaignTypedRepository,
            CampaignModifyRepository campaignModifyRepository) {
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignModifyRepository = campaignModifyRepository;
    }

    @Nullable
    @Override
    public ResetNoTitleSubstituteFlagFromCampaignsState execute(Void inputData,
                                                                ResetNoTitleSubstituteFlagFromCampaignsState prevState, int shard) {
        //если предыдущая стадия не инициализирована -- начали ваншот
        if (prevState == null) {
            logger.info("First iteration! shard: {}", shard);
            return new ResetNoTitleSubstituteFlagFromCampaignsState(0);
        }
        //получаем следующие 10000 кампаний
        List<BaseCampaign> baseCampaigns =
                campaignTypedRepository.getCampaignsWithOptionAndIdGreaterThan(
                        shard,
                        prevState.getLastCampaignId(),
                        CampaignOpts.NO_TITLE_SUBSTITUTE.getTypedValue(),
                        TYPES_WITH_OPTION,
                        LIMIT);
        List<CommonCampaign> campaigns = StreamEx.of(baseCampaigns)
                .select(CommonCampaign.class)
                .toList();

        if (baseCampaigns.isEmpty()) {
            logger.info("Last iteration! last campaign id: {}, shard: {}", prevState.getLastCampaignId(), shard);
            return null;
        }

        logger.info("last campaign id: {}, shard: {}", prevState.getLastCampaignId(), shard);
        List<AppliedChanges<CommonCampaign>> appliedChanges = StreamEx.of(campaigns)
                .map(campaign -> new ModelChanges<>(campaign.getId(), CommonCampaign.class)
                        .process(true, CommonCampaign.HAS_TITLE_SUBSTITUTION)
                        .applyTo(campaign))
                .toList();

        if (!appliedChanges.isEmpty()) {
            campaignModifyRepository.updateCampaignsTable(shard, appliedChanges);
        }

        Long lastCampaignId = StreamEx.of(baseCampaigns)
                .map(BaseCampaign::getId)
                .max(Long::compare)
                .get();
        return new ResetNoTitleSubstituteFlagFromCampaignsState(lastCampaignId);
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }
}
