package ru.yandex.direct.oneshot.oneshots.retargetingconditions

import com.google.common.collect.Lists
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService.SHORTCUT_DEFAULT_ID_BY_NAME
import ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType
import ru.yandex.direct.dbutil.QueryWithoutIndex
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.StringConstraints.notBlank
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

data class InputData(
    val oldName: String,
    val newName: String
)

/**
 * Ваншот для переименования сохраненных шорткатов.
 *
 * Имена шорткатов захардкожены и не могут быть изменены пользователями, хранятся в RetargetingConditionShortcutService.
 * При смене захардкоженых имен новые шорткаты будут создаваться с новой версией имени (и матчиться со старым именем),
 * но для стандартизации можно будет обновить имена у сохраненных шорткатов.
 *
 * Ваншот принимает на вход старое и новое имена (новое должно быть актуальным именем шортката).
 *
 * Обновляет только условия ретаргетинга с типом шорткат, поэтому пользовательские условия с такими же названиями
 * не затронет.
 */
@Component
@Multilaunch
@Approvers("elwood", "dlyange")
@PausedStatusOnFail
class RenameShortcutsOneshot @Autowired constructor(
    private val dslContextProvider: DslContextProvider
) : ShardedOneshot<InputData, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(RenameShortcutsOneshot::class.java)
        private const val RET_COND_IDS_CHUNK_SIZE = 1000
    }

    override fun validate(inputData: InputData) = validateObject(inputData) {
        property(inputData::oldName) {
            check(notNull())
            check(notBlank())
        }
        property(inputData::newName) {
            check(notNull())
            check(notBlank())
            check(
                fromPredicate(
                    { SHORTCUT_DEFAULT_ID_BY_NAME.keys.contains(it) }, objectNotFound()
                )
            )
        }
    }

    override fun execute(inputData: InputData, prevState: Void?, shard: Int): Void? {
        logger.info("shard $shard: start renaming shortcuts with name ${inputData.oldName} to ${inputData.newName}")
        val retCondIds = getRetCondIdsForUpdating(shard, inputData.oldName)
        logger.info("shard $shard: ${retCondIds.size} retargeting condition shortcuts")
        for (retCondIdsChunk in Lists.partition(retCondIds, RET_COND_IDS_CHUNK_SIZE)) {
            logger.info("shard $shard: updating $retCondIdsChunk retargeting condition shortcuts")
            updateConditionNames(shard, inputData.newName, retCondIdsChunk)
        }
        logger.info("shard $shard: finished updating retargeting condition shortcuts")
        return null
    }

    @QueryWithoutIndex("Переименование шорткатов, очень редкая ситуация")
    private fun getRetCondIdsForUpdating(shard: Int, oldName: String): List<Long> {
        return dslContextProvider.ppc(shard)
            .select(RETARGETING_CONDITIONS.RET_COND_ID)
            .from(RETARGETING_CONDITIONS)
            .where(
                RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(RetargetingConditionsRetargetingConditionsType.shortcuts)
                    .and(RETARGETING_CONDITIONS.CONDITION_NAME.eq(oldName))
            )
            .fetch(RETARGETING_CONDITIONS.RET_COND_ID)
    }

    private fun updateConditionNames(shard: Int, newName: String, retCondIds: List<Long>) {
        dslContextProvider.ppc(shard)
            .update(RETARGETING_CONDITIONS)
            .set(RETARGETING_CONDITIONS.CONDITION_NAME, newName)
            .where(RETARGETING_CONDITIONS.RET_COND_ID.`in`(retCondIds))
            .execute()
    }
}
