package ru.yandex.direct.oneshot.oneshots.returnprostrategyview;

import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSimplifiedStrategyView;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstantsService;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isAttributionModelAvailableForSimpleView;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isStrategyAvailableForSimpleView;

/**
 * Ваншот для кампаний, у которых некорректная модель атрибуции или некорректная стратегия для режима новичка,
 * выключает режим новичка на клиенте
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Approvers({"ssdmitriev", "pashkus"})
public class ReturnProStrategyViewOneshot implements ShardedOneshot<Void, ReturnProStrategyViewState> {
    private static final Logger logger = LoggerFactory.getLogger(ReturnProStrategyViewOneshot.class);

    private static final int LIMIT = 10000;

    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignModifyRepository campaignModifyRepository;
    private final CampaignConstantsService campaignConstantsService;

    @Autowired
    public ReturnProStrategyViewOneshot(CampaignTypedRepository campaignTypedRepository,
                                        CampaignModifyRepository campaignModifyRepository,
                                        CampaignConstantsService campaignConstantsService) {
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignModifyRepository = campaignModifyRepository;
        this.campaignConstantsService = campaignConstantsService;
    }


    @Nullable
    @Override
    public ReturnProStrategyViewState execute(Void inputData, ReturnProStrategyViewState prevState, int shard) {
        //если предыдущая стадия не инициализирована -- начали ваншот
        if (prevState == null) {
            logger.info("First iteration! shard: {}", shard);
            return new ReturnProStrategyViewState(0);
        }
        //получаем следующие 10000 кампаний указанных типов
        List<BaseCampaign> campaigns = campaignTypedRepository.getCampaignsForTypesAndIdGreaterThan(shard,
                prevState.getLastCampaignId(),
                Set.of(CampaignsType.dynamic, CampaignsType.text, CampaignsType.performance), LIMIT);
        if (campaigns.isEmpty()) {
            logger.info("Last iteration! last campaign id: {}, shard: {}", prevState.getLastCampaignId(), shard);
            return null;
        }

        logger.info("last campaign id: {}, shard: {}", prevState.getLastCampaignId(), shard);

        List<CommonCampaign> campaignsToEnableProView = getCampaignIdsToEnableProView(campaigns);
        List<AppliedChanges<CommonCampaign>> appliedChanges = StreamEx.of(campaignsToEnableProView)
                .map(campaign -> new ModelChanges<>(campaign.getId(), CommonCampaign.class)
                        .process(false, CommonCampaign.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED)
                        .applyTo(campaign))
                .toList();
        if (!appliedChanges.isEmpty()) {
            campaignModifyRepository.updateCampaignsTable(shard, appliedChanges);
        }
        Long lastCampaignId = StreamEx.of(campaigns)
                .map(BaseCampaign::getId)
                .max(Long::compare)
                .get();
        return new ReturnProStrategyViewState(lastCampaignId);

    }

    private List<CommonCampaign> getCampaignIdsToEnableProView(List<BaseCampaign> campaigns) {
        return StreamEx.of(campaigns)
                .select(CommonCampaign.class)
                .filter(CommonCampaign::getIsSimplifiedStrategyViewEnabled)
                .select(CampaignWithSimplifiedStrategyView.class)
                .remove(model -> isStrategyAvailableForSimpleView(model.getStrategy()) &&
                        isAttributionModelAvailableForSimpleView(model.getAttributionModel(),
                                campaignConstantsService.getDefaultAttributionModel()))
                .select(CommonCampaign.class)
                .toList();
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }

}
