package ru.yandex.direct.oneshot.oneshots.rmp_bs_tags;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;

/**
 * Ваншот для выставления тега для таргетировния на все группы указаной РМП кампании.
 * Теги отправляются в БК на группах в полях `PageGroupTags` и `TargetTags`.
 * В БД сохраняются в `adgroup_bs_tags.page_group_tags_json` и `adgroup_bs_tags.target_tags_json`.
 * <p>
 * Ожидаемый формат входных данных: `{"campaignId": 1234567, "tag": "lala tag"}`.
 * Если указать в поле `tag` значение `null`, у всех групп кампании будет удалены теги.
 */
@Approvers("hrustyashko")
@Multilaunch
@Component
@ParametersAreNonnullByDefault
public class BsTagsOnRmpOneshot implements SimpleOneshot<InputData, Void> {

    private static final Logger logger = LoggerFactory.getLogger(BsTagsOnRmpOneshot.class);
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final AdGroupRepository adGroupRepository;

    @Autowired
    public BsTagsOnRmpOneshot(
            ShardHelper shardHelper, DslContextProvider dslContextProvider,
            AdGroupRepository adGroupRepository
    ) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.adGroupRepository = adGroupRepository;
    }

    @Nullable
    @Override
    public Void execute(@Nullable InputData inputData, @Nullable Void prevState) {
        checkNotNull(inputData);
        var campaignId = inputData.getCampaignId();
        var clientId = shardHelper.getClientIdByCampaignId(campaignId);
        int shard = shardHelper.getShardByCampaignId(campaignId);
        var adGroupIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard, singletonList(campaignId));
        var adGroups = adGroupRepository.getAdGroups(shard, adGroupIds.get(campaignId));

        List<String> newTags = inputData.getTag() == null ? emptyList() : singletonList(inputData.getTag());
        ArrayList<AppliedChanges<AdGroup>> changes = new ArrayList<>();
        for (AdGroup ag : adGroups) {
            logger.info("Old tags for adGroup {}: targetTags = {}, pageGroupTags = {}",
                    ag.getId(), ag.getTargetTags(), ag.getPageGroupTags()
            );
            ModelChanges<AdGroup> mc = ModelChanges.build(ag, AdGroup.TARGET_TAGS, newTags);
            mc.process(newTags, AdGroup.PAGE_GROUP_TAGS);
            mc.process(StatusBsSynced.NO, AdGroup.STATUS_BS_SYNCED);
            changes.add(mc.applyTo(ag));
        }

        logger.info("Settings targetTags and pageGroupTags to {}", newTags);
        adGroupRepository.updateAdGroups(shard, ClientId.fromLong(clientId), changes);

        return null;
    }

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        var campaignId = inputData.getCampaignId();
        int shard = shardHelper.getShardByCampaignId(campaignId);
        if (shard == NO_SHARD) {
            logger.error("Campaign {} not found", campaignId);
            return ValidationResult.failed(inputData, CommonDefects.invalidValue());
        }
        CampaignsType campType = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.TYPE).from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.eq(campaignId)).fetchOne(CAMPAIGNS.TYPE);
        if (campType != CampaignsType.mobile_content) {
            logger.error("Campaign {} has wrong type {}", campaignId, campType);
            return ValidationResult.failed(inputData, CommonDefects.invalidValue());
        }

        return ValidationResult.success(inputData);
    }
}
