package ru.yandex.direct.oneshot.oneshots.segmentlogdate;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;

import javax.annotation.Nullable;

import com.google.common.collect.Lists;
import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.enums.VideoSegmentGoalsAudienceType;
import ru.yandex.direct.dbschema.ppc.enums.VideoSegmentGoalsExternalAudienceStatus;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.VIDEO_SEGMENT_GOALS;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidFormat;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

/**
 * Входные данные:
 * - дата в формате 2011-12-03;
 * - тип группы - такой, чтобы AdGroupType.valueOf сработал, например CPM_VIDEO
 */
@Component
@Multilaunch
@Approvers({"ovazhnev", "mexicano", "hrustyashko"})
public class SegmentLogReadDateOneshot implements ShardedOneshot<SegmentLogReadDateOneshotData, Void> {

    private static final Logger logger = LoggerFactory.getLogger(SegmentLogReadDateOneshot.class);

    private static final int CHUNK_SIZE = 1000;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public SegmentLogReadDateOneshot(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    @Nullable
    @Override
    public Void execute(SegmentLogReadDateOneshotData inputData, Void prevState, int shard) {
        logger.info("oneshot started to set last read log date {} for segments of adGroups with type ={}",
                inputData.getNewLastReadLogDate(), inputData.getAdGroupType());

        List<SegmentInfo> segmentsToUpdate = getSegmentsToUpdate(inputData, shard);
        logger.info("segments to update: {}", segmentsToUpdate.size());

        int updatedCount = updateSegments(inputData, segmentsToUpdate, shard);
        logger.info("segments updated: {}", updatedCount);

        return null;
    }

    private List<SegmentInfo> getSegmentsToUpdate(
            SegmentLogReadDateOneshotData inputData, int shard) {
        AdGroupType adGroupType = AdGroupType.valueOf(inputData.getAdGroupType());
        PhrasesAdgroupType dbAdgroupType = AdGroupType.toSource(adGroupType);

        return dslContextProvider.ppc(shard)
                .select(VIDEO_SEGMENT_GOALS.PID, VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE)
                .from(VIDEO_SEGMENT_GOALS)
                .join(PHRASES).on(PHRASES.PID.eq(VIDEO_SEGMENT_GOALS.PID))
                .where(PHRASES.ADGROUP_TYPE.eq(dbAdgroupType))
                .and(VIDEO_SEGMENT_GOALS.IS_DISABLED.eq(0L))
                .and(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_STATUS
                        .notEqual(VideoSegmentGoalsExternalAudienceStatus.deleted))
                .fetch(rec -> new SegmentInfo(rec.value1(), rec.value2()));
    }

    private int updateSegments(
            SegmentLogReadDateOneshotData inputData,
            List<SegmentInfo> segmentsToUpdate,
            int shard) {
        LocalDate newLastReadLogDate =
                LocalDate.parse(inputData.getNewLastReadLogDate(), DateTimeFormatter.ISO_LOCAL_DATE);
        LocalDateTime newLastReadLogDateTime = LocalDateTime.of(newLastReadLogDate, LocalTime.MIDNIGHT);

        AtomicInteger updatedCount = new AtomicInteger(0);
        Lists.partition(segmentsToUpdate, CHUNK_SIZE).forEach(segmentsChunk -> {

            Condition condition = DSL.falseCondition();
            for (SegmentInfo segment : segmentsChunk) {
                condition = condition.or(
                        VIDEO_SEGMENT_GOALS.PID.eq(segment.getAdGroupId())
                                .and(VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE.eq(segment.getAudienceType())));
            }

            int updatedInChunkCount = dslContextProvider.ppc(shard)
                    .update(VIDEO_SEGMENT_GOALS)
                    .set(VIDEO_SEGMENT_GOALS.LAST_SUCCESSFUL_UPDATE_TIME, newLastReadLogDateTime)
                    .where(condition)
                    .execute();

            updatedCount.addAndGet(updatedInChunkCount);
        });

        return updatedCount.get();
    }

    @Override
    public ValidationResult<SegmentLogReadDateOneshotData, Defect> validate(SegmentLogReadDateOneshotData inputData) {
        ItemValidationBuilder<SegmentLogReadDateOneshotData, Defect> builder =
                ItemValidationBuilder.of(inputData);

        builder.item(inputData.getAdGroupType(), "adGroupType")
                .check(adGroupTypeIsParsable());

        builder.item(inputData.getNewLastReadLogDate(), "newLastReadLogDate")
                .check(dateIsParsable());

        return builder.getResult();
    }

    private Constraint<String, Defect> adGroupTypeIsParsable() {
        Predicate<String> predicate = str -> {
            try {
                AdGroupType.valueOf(str);
                return true;
            } catch (Exception e) {
                return false;
            }
        };
        return Constraint.fromPredicate(predicate, invalidValue());
    }

    private Constraint<String, Defect> dateIsParsable() {
        Predicate<String> predicate = str -> {
            try {
                DateTimeFormatter.ISO_LOCAL_DATE.parse(str);
                return true;
            } catch (Exception e) {
                return false;
            }
        };
        return Constraint.fromPredicate(predicate, invalidFormat());
    }

    private static class SegmentInfo {
        final Long adGroupId;
        final VideoSegmentGoalsAudienceType audienceType;

        public SegmentInfo(Long adGroupId, VideoSegmentGoalsAudienceType audienceType) {
            this.adGroupId = adGroupId;
            this.audienceType = audienceType;
        }

        public Long getAdGroupId() {
            return adGroupId;
        }

        public VideoSegmentGoalsAudienceType getAudienceType() {
            return audienceType;
        }
    }
}
