package ru.yandex.direct.oneshot.oneshots.sendperfcamptomoderation

import com.fasterxml.jackson.annotation.JsonProperty
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.enums.*
import ru.yandex.direct.dbschema.ppc.tables.*
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

class InputData {
    @JsonProperty("campaignId")
    var campaignId: Long? = null
}

@Component
@Approvers("mspirit")
class SendPerfCampToModerationOneshot @Autowired constructor(
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper
) : SimpleOneshot<InputData, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(SendPerfCampToModerationOneshot::class.java)
    }

    override fun execute(inputData: InputData?, prevState: Void?): Void? {
        val shard = shardHelper.getShardByCampaignId(inputData!!.campaignId!!)
        dslContextProvider.ppcTransaction(shard) { cfg ->
            val dsl = cfg.dsl()

            val performanceBannersCount: Int = dsl.selectCount()
                .from(Banners.BANNERS)
                .where(Banners.BANNERS.CID.eq(inputData.campaignId))
                .and(Banners.BANNERS.BANNER_TYPE.eq(BannersBannerType.performance))
                .fetchSingle(0, Integer.TYPE)
            if (performanceBannersCount != 0) {
                logger.warn("Found performance banners in campaign, they will be ignored")
            }

            val performanceMainBanners: MutableList<Pair<Long, Long>> = dsl
                .select(Banners.BANNERS.BID, Banners.BANNERS.PID)
                .from(Banners.BANNERS)
                .where(Banners.BANNERS.CID.eq(inputData.campaignId))
                .and(Banners.BANNERS.BANNER_TYPE.eq(BannersBannerType.performance_main))
                .fetch { Pair(it[Banners.BANNERS.BID], it[Banners.BANNERS.PID]) }
            if (performanceMainBanners.size == 0) {
                logger.error("No performance_main banners in campaign")
                return@ppcTransaction
            }
            logger.info("Selected ${performanceMainBanners.size} banners: $performanceMainBanners")
            val bids = performanceMainBanners.map { it.first }.toList()
            val pids = performanceMainBanners.map { it.second }.toList()

            dsl
                .update(Campaigns.CAMPAIGNS)
                .set(Campaigns.CAMPAIGNS.STATUS_MODERATE, CampaignsStatusmoderate.Ready)
                .where(Campaigns.CAMPAIGNS.CID.eq(inputData.campaignId))
                .execute()
            dsl.update(CampOptions.CAMP_OPTIONS)
                .set(CampOptions.CAMP_OPTIONS.STATUS_POST_MODERATE, CampOptionsStatuspostmoderate.Accepted)
                .where(CampOptions.CAMP_OPTIONS.CID.eq(inputData.campaignId))
                .execute()
            dsl
                .update(Phrases.PHRASES)
                .set(Phrases.PHRASES.STATUS_SHOWS_FORECAST, PhrasesStatusshowsforecast.Sending)
                .set(Phrases.PHRASES.STATUS_MODERATE, PhrasesStatusmoderate.Yes)
                .set(Phrases.PHRASES.STATUS_POST_MODERATE, PhrasesStatuspostmoderate.Yes)
                .where(Phrases.PHRASES.CID.eq(inputData.campaignId))
                .execute()
            dsl
                .update(AdgroupsPerformance.ADGROUPS_PERFORMANCE)
                .set(
                    AdgroupsPerformance.ADGROUPS_PERFORMANCE.STATUS_BL_GENERATED,
                    AdgroupsPerformanceStatusblgenerated.Processing
                )
                .where(AdgroupsPerformance.ADGROUPS_PERFORMANCE.PID.`in`(pids))
                .execute()
            dsl.update(Banners.BANNERS)
                .set(Banners.BANNERS.STATUS_MODERATE, BannersStatusmoderate.Ready)
                .set(Banners.BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                .where(Banners.BANNERS.BID.`in`(bids))
                .execute()
        }
        logger.info("Finished")
        return null
    }

    override fun validate(inputData: InputData): ValidationResult<InputData, Defect<Any>> {
        val vb = ItemValidationBuilder.of<InputData, Defect<Any>>(inputData)
        vb.item(inputData.campaignId, "campaignId")
            .check(CommonConstraints.notNull())
        return vb.result
    }
}
