package ru.yandex.direct.oneshot.oneshots.sharded;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SafeOneshot;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.minStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Component
@Multilaunch
@SafeOneshot
public class TestShardedOneshot implements ShardedOneshot<InputData, State> {
    private static final Logger logger = LoggerFactory.getLogger(TestShardedOneshot.class);

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        ItemValidationBuilder<InputData, Defect> ivb = ItemValidationBuilder.of(inputData);
        ivb.item(inputData.getData(), "data")
                .check(notNull())
                .check(notBlank())
                .check(minStringLength(4));
        return ivb.getResult();
    }

    @Override
    public State execute(InputData inputData, State state, int shard) {
        if (state == null) {
            logger.info("First iteration! shard: {}, input data: {}", shard, inputData.getData());
            sleep();
            return new State().withIteration(1);
        }

        if (state.getIteration() >= 5) {
            logger.info("Last iteration: {}, shard: {}, input data: {}",
                    state.getIteration(), shard, inputData.getData());
            sleep();
            return null;
        } else {
            logger.info("Iteration: {}, shard: {}, input data: {}",
                    state.getIteration(), shard, inputData.getData());
            sleep();
            return state.withIteration(state.getIteration() + 1);
        }
    }

    private void sleep() {
        try {
            Thread.sleep(10_000);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }
}
